module init_g
  implicit none

  public :: ginit
  public :: init_init_g
  public :: k0
  public :: tstart
  public :: reset_init
  private

  ! knobs
  integer :: ginitopt_switch
  integer, parameter :: ginitopt_default = 1, ginitopt_test1 = 2, &
       ginitopt_zero = 3, &
       ginitopt_restart_file = 4, &
       ginitopt_noise = 5, ginitopt_restart_many = 6, ginitopt_continue = 7, &
       ginitopt_nl = 8, ginitopt_kz0 = 9, ginitopt_restart_small = 10, &
       ginitopt_nl2 = 11, ginitopt_nl3 = 12, ginitopt_nl4 = 13, & 
       ginitopt_nl5 = 14, ginitopt_alf = 15, ginitopt_kpar = 16, &
       ginitopt_nl6 = 17, ginitopt_nl7 = 18, ginitopt_gs = 19, ginitopt_recon = 20, &
       ginitopt_nl3r = 21, ginitopt_smallflat = 22, ginitopt_recon2 = 23
  real :: dphiinit, phiinit, k0, imfac, refac, zf_init
  real :: den0, upar0, tpar0, tperp0
  real :: den1, upar1, tpar1, tperp1
  real :: den2, upar2, tpar2, tperp2
  real :: tstart, scale, apar0
  logical :: chop_side, left, even
  character(300) :: restart_file
  integer, dimension(2) :: ikk, itt
  integer, dimension(3) :: ikkk, ittt
  
contains

  subroutine init_init_g
    use agk_save, only: init_save
    use agk_layouts, only: init_agk_layouts
    use mp, only: proc0, broadcast
    implicit none
    logical, save :: initialized = .false.

    if (initialized) return
    initialized = .true.

    call init_agk_layouts

    if (proc0) call read_parameters

    call broadcast (ginitopt_switch)
    call broadcast (refac)
    call broadcast (imfac)
    call broadcast (den0)
    call broadcast (upar0)
    call broadcast (tpar0)
    call broadcast (tperp0)
    call broadcast (den1)
    call broadcast (upar1)
    call broadcast (tpar1)
    call broadcast (tperp1)
    call broadcast (den2)
    call broadcast (upar2)
    call broadcast (tpar2)
    call broadcast (tperp2)
    call broadcast (phiinit)
    call broadcast (dphiinit)
    call broadcast (zf_init)
    call broadcast (k0)
    call broadcast (apar0)
    call broadcast (tstart)
    call broadcast (chop_side)
    call broadcast (even)
    call broadcast (left)
    call broadcast (restart_file)
    call broadcast (ikk)
    call broadcast (itt) 
    call broadcast (ikkk)
    call broadcast (ittt) 
    call broadcast (scale)

    call init_save (restart_file)

  end subroutine init_init_g

  subroutine ginit (restarted)

    use agk_save, only: init_tstart
! TT>
    use file_utils, only: error_unit
! <TT
    logical, intent (out) :: restarted
    real :: t0
    integer :: istatus

    restarted = .false.
    select case (ginitopt_switch)
    case (ginitopt_default)
       call ginit_default
    case (ginitopt_kz0)
       call ginit_kz0
    case (ginitopt_noise)
       call ginit_noise
    case (ginitopt_kpar)
       call ginit_kpar
    case (ginitopt_gs)
       call ginit_gs
! TT> added for ES decaying turbulence problem
    case (ginitopt_nl)
       call ginit_nl
! TT: added for Orszag-Tang vortex problem
    case (ginitopt_nl2)
       call ginit_nl2
! <TT
    case (ginitopt_nl3r)
       call ginit_nl3r
    case (ginitopt_nl4)
       call ginit_nl4
       call init_tstart (tstart, istatus)
       restarted = .true.
       scale = 1.
    case (ginitopt_recon)
       t0 = tstart
       call init_tstart (tstart, istatus)
       call ginit_recon
       tstart = t0
       restarted = .true.
       scale = 1.
    case (ginitopt_recon2)
       t0 = tstart
       call init_tstart (tstart, istatus)
       call ginit_recon2
       tstart = t0
       restarted = .true.
       scale = 1.
    case (ginitopt_alf)
       call ginit_alf
    case (ginitopt_zero)
       call ginit_zero
    case (ginitopt_restart_file)
       call ginit_restart_file 
       call init_tstart (tstart, istatus)
       restarted = .true.
       scale = 1.
    case (ginitopt_restart_many)
       call ginit_restart_many 
       call init_tstart (tstart, istatus)
       restarted = .true.
       scale = 1.
    case (ginitopt_restart_small)
       call ginit_restart_small
       call init_tstart (tstart, istatus)
       restarted = .true.
       scale = 1.
    case (ginitopt_smallflat)
       call ginit_restart_smallflat
       call init_tstart (tstart, istatus)
       restarted = .true.
       scale = 1.
    case (ginitopt_continue)
       restarted = .true.
       scale = 1.
! TT> added error message
    case default
       write (error_unit(),*) 'WARNING: No initial condition subroutine found'
! <TT
    end select
  end subroutine ginit

  subroutine read_parameters
    use file_utils, only: input_unit, error_unit, run_name, input_unit_exist
    use text_options
    implicit none

    type (text_option), dimension (23), parameter :: ginitopts = &
         (/ text_option('default', ginitopt_default), &
            text_option('noise', ginitopt_noise), &
            text_option('test1', ginitopt_test1), &
            text_option('zero', ginitopt_zero), &
            text_option('many', ginitopt_restart_many), &
            text_option('small', ginitopt_restart_small), &
            text_option('file', ginitopt_restart_file), &
            text_option('cont', ginitopt_continue), &
            text_option('kz0', ginitopt_kz0), &
            text_option('nl', ginitopt_nl), &
            text_option('nl2', ginitopt_nl2), &
            text_option('nl3', ginitopt_nl3), &
            text_option('nl3r', ginitopt_nl3r), &
            text_option('nl4', ginitopt_nl4), &
            text_option('nl5', ginitopt_nl5), &
            text_option('nl6', ginitopt_nl6), &
            text_option('nl7', ginitopt_nl7), &
            text_option('alf', ginitopt_alf), &
            text_option('gs', ginitopt_gs), &
            text_option('kpar', ginitopt_kpar), &
            text_option('smallflat', ginitopt_smallflat), &
            text_option('recon', ginitopt_recon), &
            text_option('recon2', ginitopt_recon2) /)
    character(20) :: ginit_option
    namelist /init_g_knobs/ ginit_option, phiinit, k0, chop_side, &
         restart_file, left, ikk, itt, scale, tstart, zf_init, &
         den0, upar0, tpar0, tperp0, imfac, refac, even, &
         den1, upar1, tpar1, tperp1, &
         den2, upar2, tpar2, tperp2, dphiinit, apar0, ikkk, ittt

    integer :: ierr, in_file
    logical :: exist

    tstart = 0.
    scale = 1.0
    ginit_option = "default"
    refac = 1.
    imfac = 0.
    den0 = 1.
    upar0 = 0.
    tpar0 = 0.
    tperp0 = 0.
    den1 = 0.
    upar1 = 0.
    tpar1 = 0.
    tperp1 = 0.
    den2 = 0.
    upar2 = 0.
    tpar2 = 0.
    tperp2 = 0.
    phiinit = 1.0
    dphiinit = 1.0
    zf_init = 1.0
    k0 = 1.0
    apar0 = 0.
    chop_side = .true.
    left = .true.
    even = .true.
    ikk(1) = 1
    ikk(2) = 2
    itt(1) = 1
    itt(2) = 2
    ikkk(1) = 1
    ikkk(2) = 2
    ikkk(3) = 2
    ittt(1) = 1
    ittt(2) = 2
    ittt(3) = 2
    restart_file = trim(run_name)//".nc"
    in_file = input_unit_exist ("init_g_knobs", exist)
    if (exist) read (unit=input_unit("init_g_knobs"), nml=init_g_knobs)

    ierr = error_unit()
    call get_option_value &
         (ginit_option, ginitopts, ginitopt_switch, &
         ierr, "ginit_option in ginit_knobs")
  end subroutine read_parameters

  subroutine ginit_default
    use species, only: spec
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, aky, reality
    use dist_fn_arrays, only: g, gnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi
    logical :: right
    integer :: iglo
    integer :: ig, ik, it, il, is

    right = .not. left

    if (chop_side .and. left) phi(:-1,:,:) = 0.0
    if (chop_side .and. right) phi(1:,:,:) = 0.0
    
    if (reality) then
       phi(:,1,1) = 0.0

       if (naky > 1 .and. aky(1) == 0.0) then
          phi(:,:,1) = 0.0
       end if

! not used:
! reality condition for k_theta = 0 component:
       do it = 1, nakx/2
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
       enddo
    end if

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       g(:,1,iglo) = phi(:,it,ik)*spec(is)%z*phiinit
       g(:,2,iglo) = g(:,1,iglo)
    end do
    gnew = g
  end subroutine ginit_default

  subroutine ginit_kz0
    use species, only: spec
    use theta_grid, only: ntgrid 
    use kgrids, only: naky, nakx, aky, reality
    use dist_fn_arrays, only: g, gnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi
    logical :: right
    integer :: iglo
    integer :: ik, it, il, is

    right = .not. left

    phi = cmplx(1.0,1.0)
    if (chop_side .and. left) phi(:-1,:,:) = 0.0
    if (chop_side .and. right) phi(1:,:,:) = 0.0
    
    if (reality) then
       phi(:,1,1) = 0.0

       if (naky > 1 .and. aky(1) == 0.0) then
          phi(:,:,1) = 0.0
       end if

! not used:
! reality condition for k_theta = 0 component:
       do it = 1, nakx/2
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
       enddo
    end if

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       g(:,1,iglo) = -phi(:,it,ik)*spec(is)%z*phiinit
       g(:,2,iglo) = g(:,1,iglo)
    end do
    gnew = g
  end subroutine ginit_kz0

  subroutine ginit_noise
    use species, only: spec, tracer_species
    use theta_grid, only: ntgrid 
    use kgrids, only: naky, nakx, aky, reality
    use dist_fn_arrays, only: g, gnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use ran
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi, phit
    real :: a, b
    integer :: iglo
    integer :: ig, ik, it, il, is, nn

    phit = 0.
    do it=2,nakx/2+1
       nn = it-1
! extra factor of 4 to reduce the high k wiggles for now
       phit (:, it, 1) = (-1)**nn*exp(-8.*(real(nn)/nakx)**2)
    end do

! keep old (it, ik) loop order to get old results exactly: 
    do it = 1, nakx
       do ik = 1, naky
          do ig = -ntgrid, ntgrid
             a = ranf()-0.5
             b = ranf()-0.5
!             phi(:,it,ik) = cmplx(a,b)
             phi(ig,it,ik) = cmplx(a,b)
           end do
          if (chop_side) then
             if (left) then
                phi(:-1,it,ik) = 0.0
             else
                phi(1:,it,ik) = 0.0
             end if
          end if
       end do
    end do

    if (naky > 1 .and. aky(1) == 0.0) then
       phi(:,:,1) = phi(:,:,1)*zf_init
    end if
! reality condition for k_theta = 0 component:
    if (reality) then
       do it = 1, nakx/2
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
          phit(:,it+(nakx+1)/2,1) = conjg(phit(:,(nakx+1)/2+1-it,1))
       enddo
    end if
       
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       if (spec(is)%type == tracer_species) then          
          g(:,1,iglo) =-phit(:,it,ik)*spec(is)%z*phiinit
       else
          g(:,1,iglo) = -phi(:,it,ik)*spec(is)%z*phiinit
       end if
       g(:,2,iglo) = g(:,1,iglo)
    end do
    gnew = g

  end subroutine ginit_noise

! TT> made up for decaying turbulence simulation
  subroutine ginit_nl
    use species, only: spec
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, reality
    use dist_fn_arrays, only: g, gnew, vpa, vperp2
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use constants, only: zi
    use ran, only: ranf
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi, odd
    real, dimension (-ntgrid:ntgrid) :: dfac, ufac, tparfac, tperpfac, ct, st, c2t, s2t
    integer :: iglo
    integer :: ig, ik, it, il, is, j
    
    phi = 0.
    odd = 0.
    do j = 1, 3
       ik = ikkk(j)
       it = ittt(j)
       phi(:,it,ik) = cmplx(ranf()-.5, ranf()-.5) ! homogeneous along z
    end do

    odd = zi * phi

! reality condition for kx = 0 component:
    if (reality) then
       do it = 1, nakx/2
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
          odd(:,it+(nakx+1)/2,1) = conjg(odd(:,(nakx+1)/2+1-it,1))
       end do
    end if

    if (even) then
       ct = cos(theta)
       st = sin(theta)

       c2t = cos(2.*theta)
       s2t = sin(2.*theta)
    else
       ct = sin(theta)
       st = cos(theta)

       c2t = sin(2.*theta)
       s2t = cos(2.*theta)
    end if

    dfac     = den0   + den1 * ct + den2 * c2t
    ufac     = upar0  + upar1* st + upar2* s2t
    tparfac  = tpar0  + tpar1* ct + tpar2* c2t
    tperpfac = tperp0 + tperp1*ct + tperp2*c2t


! charge dependence keeps initial Phi from being too small
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)       
       g(:,1,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0                * phi(:,it,ik) &
            + 2.*ufac* vpa(1,iglo)*spec(is)%u0 * odd(:,it,ik) &
            + tparfac*(vpa(1,iglo)**2-0.5)     * phi(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)        * phi(:,it,ik))

       g(:,2,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0                * phi(:,it,ik) &
            + 2.*ufac* vpa(2,iglo)*spec(is)%u0 * odd(:,it,ik) &
            + tparfac*(vpa(2,iglo)**2-0.5)     * phi(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)        * phi(:,it,ik))

    end do

    gnew = g
  end subroutine ginit_nl

  ! Orszag-Tang 2D vortex problem
  subroutine ginit_nl2
    use species, only: spec, has_electron_species, electron_species
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, reality
    use dist_fn_arrays, only: g, gnew, vpa
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use constants, only: pi
    implicit none
    integer :: iglo
    integer :: ig, ik, it, il, is
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi, jpar !! local !!
    real, dimension (-ntgrid:ntgrid) :: dfac, ufac, tparfac, tperpfac, ct, st, c2t, s2t

    !! phi, jpar are local !!
    phi = 0.0 ; jpar = 0.0
    phi(:,1,2) = phiinit * cmplx(2.0, 0.0)  ! 2 cos(y)
    phi(:,2,1) = phiinit * cmplx(1.0, 0.0)  ! 2 cos(x)
    jpar(:,1,2) = apar0 * cmplx(2.0, 0.0) ! 2 cos(y)
    jpar(:,3,1) = apar0 * cmplx(2.0, 0.0) ! 4 cos(2x)

! reality condition for ky = 0 component:
    if (reality) then
       do it = 1, nakx/2
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
          jpar(:,it+(nakx+1)/2,1) = conjg(jpar(:,(nakx+1)/2+1-it,1))
       end do
    end if

    if (even) then
       ct = cos(theta)
       st = sin(theta)

       c2t = cos(2.*theta)
       s2t = sin(2.*theta)
    else
       ct = sin(theta)
       st = cos(theta)

       c2t = sin(2.*theta)
       s2t = cos(2.*theta)
    end if

    dfac     = den0   + den1 * ct + den2 * c2t
    ufac     = upar0  + upar1* st + upar2* s2t
    tparfac  = tpar0  + tpar1* ct + tpar2* c2t
    tperpfac = tperp0 + tperp1*ct + tperp2*c2t

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)

       g(:,1,iglo) = &
            ( dfac*spec(is)%dens0                * phi(:,it,ik) &
            + 2.*ufac* vpa(1,iglo)*spec(is)%u0 * jpar(:,it,ik) )!&
!            + tparfac*(vpa(1,iglo)**2-0.5)     * phi(:,it,ik) &
!            +tperpfac*(vperp2(iglo)-1.)        * phi(:,it,ik))

       g(:,2,iglo) = &
            ( dfac*spec(is)%dens0                * phi(:,it,ik) &
            + 2.*ufac* vpa(2,iglo)*spec(is)%u0 * jpar(:,it,ik) )!&
!            + tparfac*(vpa(2,iglo)**2-0.5)     * phi(:,it,ik) &
!            +tperpfac*(vperp2(iglo)-1.)        * phi(:,it,ik))

    end do

    gnew = g

  end subroutine ginit_nl2
! <TT

  subroutine ginit_nl3r
    use species, only: spec, has_electron_species, electron_species
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, reality
    use fields_arrays, only: apar, bpar
    use fields_arrays, only: aparnew, bparnew
    use dist_fn_arrays, only: g, gnew, vpa, vperp2
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use constants
    use ran
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi, odd
    real, dimension (-ntgrid:ntgrid) :: dfac, ufac, tparfac, tperpfac, ct, st, c2t, s2t
    integer :: iglo
    integer :: ig, ik, it, il, is, j
    
    phi = 0.
    odd = 0.
    do j = 1, 2
       ik = ikk(j)
       it = itt(j)
       do ig = -ntgrid, ntgrid
          phi(ig,it,ik) = cmplx(refac, imfac)
          apar(ig,it,ik) = apar0*cmplx(refac, imfac)
       end do
       if (chop_side) then
          if (left) then
             phi(:-1,it,ik) = 0.0
          else
             phi(1:,it,ik) = 0.0
          end if
       end if
    end do

    odd = zi * phi
    
! reality condition for k_theta = 0 component:
    if (reality) then
       do it = 1, nakx/2
          apar(:,it+(nakx+1)/2,1) = conjg(apar(:,(nakx+1)/2+1-it,1))
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
          odd(:,it+(nakx+1)/2,1) = conjg(odd(:,(nakx+1)/2+1-it,1))
       enddo
    end if

    aparnew = apar

    if (even) then
       ct = cos(theta)
       st = sin(theta)

       c2t = cos(2.*theta)
       s2t = sin(2.*theta)
    else
       ct = sin(theta)
       st = cos(theta)

       c2t = sin(2.*theta)
       s2t = cos(2.*theta)
    end if

    dfac     = den0   + den1 * ct + den2 * c2t
    ufac     = upar0  + upar1* st + upar2* s2t
    tparfac  = tpar0  + tpar1* ct + tpar2* c2t
    tperpfac = tperp0 + tperp1*ct + tperp2*c2t


! charge dependence keeps initial Phi from being too small
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)       
       g(:,1,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0                * phi(:,it,ik) &
            + 2.*ufac* vpa(1,iglo)*spec(is)%u0 * odd(:,it,ik) &
            + tparfac*(vpa(1,iglo)**2-0.5)     * phi(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)        * phi(:,it,ik))

       g(:,2,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0                * phi(:,it,ik) &
            + 2.*ufac* vpa(2,iglo)*spec(is)%u0 * odd(:,it,ik) &
            + tparfac*(vpa(2,iglo)**2-0.5)     * phi(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)        * phi(:,it,ik))

    end do

!    if (has_electron_species(spec)) then
!       call flae (g, gnew)
!       g = g - gnew
!    end if

    gnew = g
  end subroutine ginit_nl3r

  subroutine ginit_recon2
    use mp, only: proc0
    use species, only: spec, has_electron_species
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, reality
    use agk_save, only: agk_restore
    use dist_fn_arrays, only: g, gnew, vpa, vperp2
    use fields_arrays, only: phi, apar, bpar
    use fields_arrays, only: phinew, aparnew, bparnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use file_utils, only: error_unit
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use constants
    use ran
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phiz, odd
    real, dimension (-ntgrid:ntgrid) :: dfac, ufac, tparfac, tperpfac, ct, st, c2t, s2t
    integer :: iglo, istatus, ierr
    integer :: ig, ik, it, il, is, j
    logical :: many = .true.
    
    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar, many)
    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       if (ik == 2) cycle

       g (:,1,iglo) = 0.
       g (:,2,iglo) = 0.
    end do
	
    phinew(:,:,1) = 0.
    aparnew(:,:,1) = 0.
    bparnew(:,:,1) = 0.
    phi(:,:,1) = 0.
    apar(:,:,1) = 0.
    bpar(:,:,1) = 0.

    phinew(:,:,3:naky) = 0.
    aparnew(:,:,3:naky) = 0.
    bparnew(:,:,3:naky) = 0.
    phi(:,:,3:naky) = 0.
    apar(:,:,3:naky) = 0.
    bpar(:,:,3:naky) = 0.

    phiz = 0.
    odd = 0.
    do j = 1, 2
       ik = ikk(j)
       it = itt(j)
       do ig = -ntgrid, ntgrid
          phiz(ig,it,ik) = cmplx(refac, imfac)
       end do
    end do

    odd = zi * phiz
    
! reality condition for k_theta = 0 component:
    if (reality) then
       do it = 1, nakx/2
          phiz(:,it+(nakx+1)/2,1) = conjg(phiz(:,(nakx+1)/2+1-it,1))
          odd (:,it+(nakx+1)/2,1) = conjg(odd (:,(nakx+1)/2+1-it,1))
       enddo
    end if

    if (even) then
       ct = cos(theta)
       st = sin(theta)

       c2t = cos(2.*theta)
       s2t = sin(2.*theta)
    else
       ct = sin(theta)
       st = cos(theta)

       c2t = sin(2.*theta)
       s2t = cos(2.*theta)
    end if

    dfac     = den0   + den1 * ct + den2 * c2t
    ufac     = upar0  + upar1* st + upar2* s2t
    tparfac  = tpar0  + tpar1* ct + tpar2* c2t
    tperpfac = tperp0 + tperp1*ct + tperp2*c2t


! charge dependence keeps initial Phi from being too small
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)       

       g(:,1,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0            * phiz(:,it,ik) &
            + 2.*ufac* vpa(1,iglo)         * odd (:,it,ik) &
            + tparfac*(vpa(1,iglo)**2-0.5) * phiz(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)    * phiz(:,it,ik))

       g(:,2,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0            * phiz(:,it,ik) &
            + 2.*ufac* vpa(2,iglo)         * odd (:,it,ik) &
            + tparfac*(vpa(2,iglo)**2-0.5) * phiz(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)    * phiz(:,it,ik))

    end do

!    if (has_electron_species(spec)) then
!       call flae (g, gnew)
!       g = g - gnew
!    end if

    gnew = g
  end subroutine ginit_recon2

  subroutine ginit_recon
    use mp, only: proc0
    use species, only: spec, has_electron_species
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, reality
    use agk_save, only: agk_restore
    use dist_fn_arrays, only: g, gnew, vpa, vperp2
    use fields_arrays, only: phi, apar, bpar
    use fields_arrays, only: phinew, aparnew, bparnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use file_utils, only: error_unit
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use constants
    use ran
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phiz, odd
    real, dimension (-ntgrid:ntgrid) :: dfac, ufac, tparfac, tperpfac, ct, st, c2t, s2t
    real :: kfac
    integer :: iglo, istatus, ierr
    integer :: ig, ik, it, il, is, j
    logical :: many = .true.
    
!
! hard-wiring some changes for a test run.  7/13/05
!

    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar, many)
    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       if (ik == 1) cycle

       g (:,1,iglo) = 0.
       g (:,2,iglo) = 0.
    end do
	
    phinew(:,:,2:naky) = 0.
    aparnew(:,:,2:naky) = 0.
    bparnew(:,:,2:naky) = 0.
    phi(:,:,2:naky) = 0.
    apar(:,:,2:naky) = 0.
    bpar(:,:,2:naky) = 0.

    phiz = 0.
    odd = 0.
    kfac = 1.
    do j = 1, 3
       ik = ikkk(j)
       it = ittt(j)
       if (j == 2) kfac =  0.5
       if (j == 3) kfac = -0.5
       do ig = -ntgrid, ntgrid
          phiz(ig,it,ik) = cmplx(refac, imfac)*kfac
       end do
    end do

    odd = zi * phiz
    
! reality condition for k_theta = 0 component:
    if (reality) then
       do it = 1, nakx/2
          phiz(:,it+(nakx+1)/2,1) = conjg(phiz(:,(nakx+1)/2+1-it,1))
          odd (:,it+(nakx+1)/2,1) = conjg(odd (:,(nakx+1)/2+1-it,1))
       enddo
    end if

    if (even) then
       ct = cos(theta)
       st = sin(theta)

       c2t = cos(2.*theta)
       s2t = sin(2.*theta)
    else
       ct = sin(theta)
       st = cos(theta)

       c2t = sin(2.*theta)
       s2t = cos(2.*theta)
    end if

    dfac     = den0   + den1 * ct + den2 * c2t
    ufac     = upar0  + upar1* st + upar2* s2t
    tparfac  = tpar0  + tpar1* ct + tpar2* c2t
    tperpfac = tperp0 + tperp1*ct + tperp2*c2t


! charge dependence keeps initial Phi from being too small
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)       

       g(:,1,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0            * phiz(:,it,ik) &
            + 2.*ufac* vpa(1,iglo)         * odd (:,it,ik) &
            + tparfac*(vpa(1,iglo)**2-0.5) * phiz(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)    * phiz(:,it,ik))

       g(:,2,iglo) = phiinit* &!spec(is)%z* &
            ( dfac*spec(is)%dens0            * phiz(:,it,ik) &
            + 2.*ufac* vpa(2,iglo)         * odd (:,it,ik) &
            + tparfac*(vpa(2,iglo)**2-0.5) * phiz(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)    * phiz(:,it,ik))

    end do

!    if (has_electron_species(spec)) then
!       call flae (g, gnew)
!       g = g - gnew
!    end if

    gnew = g
  end subroutine ginit_recon

! TT> imported from GS2 for recon run
! nl4 has been monkeyed with over time.  Do not expect to recover old results
! that use this startup routine.
  subroutine ginit_nl4
    use mp, only: proc0
    use species, only: spec
    use agk_save, only: agk_restore
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    use dist_fn_arrays, only: g, gnew
! TT>
!    use le_grids, only: forbid
! <TT
    use fields_arrays, only: phi, apar, bpar
    use fields_arrays, only: phinew, aparnew, bparnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, is_idx, il_idx
    use file_utils, only: error_unit
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use ran, only: ranf
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phiz
    integer :: iglo, istatus
    integer :: ig, ik, it, is, il, ierr
    logical :: many = .true.
    
    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar, many)
    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       it = it_idx(g_lo,iglo)
       ik = ik_idx(g_lo,iglo)
!       if (ik == 1) cycle
! TT>
!       if (it == 1 .and. ik == 2) cycle
!       if ((it == 2 .or. it == ntheta0) .and. ik == 1) cycle
       if ((it == 2 .or. it == nakx) .and. ik == 1) cycle
! <TT

       g (:,1,iglo) = 0.
       g (:,2,iglo) = 0.
    end do

    do ik = 1, naky
! TT>
!       do it=1, ntheta0
!          if (it == 1 .and. ik == 2) cycle
!          if ((it == 2 .or. it == ntheta0) .and. ik == 1) cycle
       do it=1, nakx
          if ((it == 2 .or. it == nakx) .and. ik == 1) cycle
! <TT
          phinew(:,it,ik) = 0.
          aparnew(:,it,ik) = 0.
          bparnew(:,it,ik) = 0.
          phi(:,it,ik) = 0.
          apar(:,it,ik) = 0.
          bpar(:,it,ik) = 0.          
       end do
    end do
    
    do ik = 1, naky
       do it = 1, nakx
          do ig = -ntgrid, ntgrid
             phiz(ig,it,ik) = cmplx(ranf()-0.5,ranf()-0.5)
          end do
          if (chop_side) then
             if (left) then
                phiz(:-1,it,ik) = 0.0
             else
                phiz(1:,it,ik) = 0.0
             end if
          end if
       end do
    end do

! reality condition for k_theta = 0 component:
    do it = 1, nakx/2
       phiz(:,it+(nakx+1)/2,1) = conjg(phiz(:,(nakx+1)/2+1-it,1))
    end do

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       g(:,1,iglo) = g(:,1,iglo)-phiz(:,it,ik)*spec(is)%z*phiinit
       g(:,2,iglo) = g(:,2,iglo)-phiz(:,it,ik)*spec(is)%z*phiinit
! TT>
!       where (forbid(:,il)) g(:,1,iglo) = 0.0
!       where (forbid(:,il)) g(:,2,iglo) = 0.0
! <TT
    end do
    gnew = g

  end subroutine ginit_nl4

  subroutine ginit_kpar
    use species, only: spec, has_electron_species
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx
    use dist_fn_arrays, only: g, gnew, vpa, vperp2
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use constants
    use ran
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi, odd
    real, dimension (-ntgrid:ntgrid) :: dfac, ufac, tparfac, tperpfac
    integer :: iglo
    integer :: ig, ik, it, il, is, j
    
    phi = 0.
    odd = 0.
    do ig = -ntgrid, ntgrid
       phi(ig,:,:) = cmplx(refac, imfac)
    end do
    if (chop_side) then
       if (left) then
          phi(:-1,:,:) = 0.0
       else
          phi(1:,:,:) = 0.0
       end if
    end if

    odd = zi * phi
        
    dfac     = den0   + den1 * cos(theta) + den2 * cos(2.*theta) 
    ufac     = upar0  + upar1* sin(theta) + upar2* sin(2.*theta) 
    tparfac  = tpar0  + tpar1* cos(theta) + tpar2* cos(2.*theta) 
    tperpfac = tperp0 + tperp1*cos(theta) + tperp2*cos(2.*theta) 

! charge dependence keeps initial Phi from being too small
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)       

       g(:,1,iglo) = phiinit* &!spec(is)%z* &
            ( dfac                           * phi(:,it,ik) &
            + 2.*ufac* vpa(1,iglo)         * odd(:,it,ik) &
            + tparfac*(vpa(1,iglo)**2-0.5) * phi(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)    * phi(:,it,ik))

       g(:,2,iglo) = phiinit* &!spec(is)%z* &
            ( dfac                           * phi(:,it,ik) &
            + 2.*ufac* vpa(2,iglo)         * odd(:,it,ik) &
            + tparfac*(vpa(2,iglo)**2-0.5) * phi(:,it,ik) &
            +tperpfac*(vperp2(iglo)-1.)    * phi(:,it,ik))

    end do

    if (has_electron_species(spec)) then
       call flae (g, gnew)
       g = g - gnew
    end if

    gnew = g
  end subroutine ginit_kpar

  subroutine ginit_gs
    use species, only: spec, has_electron_species
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, akx, reality
    use dist_fn_arrays, only: g, gnew, vpa, vperp2
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use constants
    use ran
    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi, odd
    integer :: iglo
    integer :: ig, ik, it, il, is
    real :: phase
    
    phi = 0.
    odd = 0.
    do ik=1,naky
       do it=1,nakx
          phase = 2.*pi*ranf()
          phi(:,it,ik) = cos(theta+phase)*cmplx(refac,imfac)
          odd(:,it,ik) = sin(theta+phase)*cmplx(refac,imfac) * zi
       end do
    end do
    
! reality condition for k_theta = 0 component:
    do it = 1, nakx/2
       phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
       odd(:,it+(nakx+1)/2,1) = conjg(odd(:,(nakx+1)/2+1-it,1))
    enddo

! charge dependence keeps initial Phi from being too small
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)       

       g(:,1,iglo) = phiinit* &!spec(is)%z* &
            ( den1                         * phi(:,it,ik) &
            + 2.*upar1* vpa(1,iglo)      * odd(:,it,ik) &
            + tpar1*(vpa(1,iglo)**2-0.5) * phi(:,it,ik) &
            + tperp1*(vperp2(iglo)-1.)   * phi(:,it,ik))
       
       g(:,2,iglo) = phiinit* &!spec(is)%z* &
            ( den1                         * phi(:,it,ik) &
            + 2.*upar1* vpa(2,iglo)      * odd(:,it,ik) &
            + tpar1*(vpa(2,iglo)**2-0.5) * phi(:,it,ik) &
            + tperp1*(vperp2(iglo)-1.)   * phi(:,it,ik))
    end do

    if (has_electron_species(spec)) then
       call flae (g, gnew)
       g = g - gnew
    end if

    gnew = g
  end subroutine ginit_gs

  subroutine ginit_alf
    use theta_grid, only: theta
    use dist_fn_arrays, only: g, gnew, vpa
    use agk_layouts, only: g_lo, il_idx, is_idx
    use species, only: spec, electron_species

    implicit none
    integer :: iglo
    integer :: il, is

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo,iglo)
       if (spec(is_idx(g_lo, iglo))%type == electron_species) cycle
       il = il_idx(g_lo,iglo)
       g(:,1,iglo) = sin(theta)*vpa(1,iglo)*spec(is)%z
       g(:,2,iglo) = sin(theta)*vpa(2,iglo)*spec(is)%z
    end do
    g = phiinit * g 
    gnew = g
  end subroutine ginit_alf

  subroutine ginit_zero
    use dist_fn_arrays, only: g, gnew
    implicit none
    g = 0.0
    gnew = 0.0
  end subroutine ginit_zero

  subroutine ginit_restart_file
    use dist_fn_arrays, only: g, gnew
    use agk_save, only: agk_restore
    use mp, only: proc0
    use file_utils, only: error_unit
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    implicit none
    integer :: istatus, ierr

    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar)
    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if
    gnew = g

  end subroutine ginit_restart_file

  subroutine ginit_restart_many
    use dist_fn_arrays, only: g, gnew
    use agk_save, only: agk_restore
    use mp, only: proc0
    use file_utils, only: error_unit
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    implicit none
    integer :: istatus, ierr
    logical :: many = .true.

    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar, many)

    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if
    gnew = g

  end subroutine ginit_restart_many

  subroutine ginit_restart_small
    use dist_fn_arrays, only: g, gnew
    use agk_save, only: agk_restore
    use mp, only: proc0
    use file_utils, only: error_unit
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    implicit none
    integer :: istatus, ierr
    logical :: many = .true.

    call ginit_noise

    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar, many)
    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if
    g = g + gnew
    gnew = g 

  end subroutine ginit_restart_small

  subroutine ginit_restart_smallflat

    use agk_save, only: agk_restore
    use mp, only: proc0
    use file_utils, only: error_unit
    use species, only: spec
    use theta_grid, only: ntgrid 
    use kgrids, only: naky, nakx, aky, reality
    use dist_fn_arrays, only: g, gnew
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, is_idx
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use ran

    implicit none
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: phi
    integer :: istatus, ierr
    logical :: many = .true.
    real :: a, b
    integer :: iglo
    integer :: ig, ik, it, il, is

    do it = 1, nakx
       do ik = 1, naky
          a = ranf()-0.5
          b = ranf()-0.5
          phi(:,it,ik) = cmplx(a,b)
       end do
    end do

    if (naky > 1 .and. aky(1) == 0.0) then
       phi(:,:,1) = phi(:,:,1)*zf_init
    end if
! reality condition for k_theta = 0 component:
    if (reality) then
       do it = 1, nakx/2
          phi(:,it+(nakx+1)/2,1) = conjg(phi(:,(nakx+1)/2+1-it,1))
       enddo
    end if

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       g(:,1,iglo) = -phi(:,it,ik)*spec(is)%z*phiinit
       g(:,2,iglo) = g(:,1,iglo)
    end do
    gnew = g

    call agk_restore (g, scale, istatus, use_Phi, use_Apar, use_Bpar, many)
    if (istatus /= 0) then
       ierr = error_unit()
       if (proc0) write(ierr,*) "Error reading file: ", trim(restart_file)
       g = 0.
    end if
    g = g + gnew
    gnew = g 

  end subroutine ginit_restart_smallflat

  subroutine reset_init

    ginitopt_switch = ginitopt_restart_many

  end subroutine reset_init

  subroutine flae (g, gavg)
    use species, only: spec, electron_species 
    use theta_grid, only: ntgrid, delthet, jacob
    use kgrids, only: aky
    use agk_layouts, only: g_lo, is_idx, ik_idx
    complex, dimension (-ntgrid:,:,g_lo%llim_proc:), intent (in) :: g
    complex, dimension (-ntgrid:,:,g_lo%llim_proc:), intent (out) :: gavg

    real :: wgt
    integer :: iglo
    
    gavg = 0.
    wgt = 1./(delthet*jacob)

    do iglo = g_lo%llim_proc, g_lo%ulim_proc       
       if (spec(is_idx(g_lo, iglo))%type /= electron_species) cycle
       if (aky(ik_idx(g_lo, iglo)) /= 0.) cycle
       gavg(:,1,iglo) = sum(g(:,1,iglo)*delthet*jacob)*wgt
       gavg(:,2,iglo) = sum(g(:,2,iglo)*delthet*jacob)*wgt
    end do

  end subroutine flae

end module init_g

