module kgrids
  implicit none

  public :: init_kgrids   ! procedure to initialize this module
  public :: aky, akx, akperp  ! (ky, kx, k_perp) * rho_ref
  public :: naky, nakx   ! #s of spectral modes
  public :: nx, ny       ! #s of grid points
  public :: x0, y0       ! Box length [x,y] = 2 * pi * [x,y]0
  public :: nkpolar      ! for diagnostics of fluctuation spectra
  public :: ikx, iky     ! map from indices of k-arrays to k-values.
  public :: box          ! true if calculation is in 2-D domain.  Otherwise running linearly for single mode.
  public :: reality      ! true if reality condition should be enforced (for box domains)
  private

  real :: akperp, x0, y0
  real, dimension (:), allocatable :: akx, aky
  integer :: naky, nakx, nx, ny, nkpolar
  integer, dimension(:), allocatable :: ikx, iky
  logical :: reality = .false.
  logical :: box = .false.

  ! internal variables
  integer :: gridopt_switch
  integer, parameter :: gridopt_single = 1, gridopt_box = 2

contains

  subroutine init_kgrids
    use mp, only: proc0, broadcast
    use file_utils, only: input_unit, error_unit, input_unit_exist
    use text_options
    implicit none

    type (text_option), dimension (3), parameter :: gridopts = &
         (/ text_option('default', gridopt_single), &
            text_option('single', gridopt_single), &
            text_option('box', gridopt_box) /)
    character(20) :: grid_option
    logical :: initialized = .false.

    namelist /kgrids/ grid_option, akperp, x0, y0, nx, ny, nkpolar
    integer :: ierr, in_file, i
    logical :: exist
    
    if (initialized) return
    initialized = .true.

! Set defaults
    nkpolar = 0
    akperp = 0.4
    nakx = 0
    naky = 0
    nx = 0
    ny = 0

    x0 = 10.
    y0 = 10.

    if (proc0) then  ! Processor 0 reads namelist 
       grid_option = 'default'
       in_file = input_unit_exist ("kgrids", exist)
       if (exist) read (unit=input_unit("kgrids"), nml=kgrids)
       
       ierr = error_unit()
       call get_option_value (grid_option, gridopts, gridopt_switch, &
            ierr, "grid_option in kgrids")
       
       select case (gridopt_switch)
       case (gridopt_single)
          
          nakx = 1 ; naky = 1 ; nx = 0 ; ny = 0
          
       case (gridopt_box)
          
          if (y0 < 0) y0 = -1./y0
          if (x0 < 0) x0 = -1./x0

          naky = (ny-1)/3 + 1            ! Allow for dealiasing 
          nakx = 2*((nx-1)/3) + 1        ! Allow for dealiasing

          if (nkpolar == 0) nkpolar = int(real(naky-1.)*sqrt(2.))

          reality = .true.
          box = .true.

       end select
    end if

! Share input data among processors
    call broadcast (nx)
    call broadcast (ny)
    call broadcast (nakx)
    call broadcast (naky)
    call broadcast (x0)
    call broadcast (y0)
    call broadcast (akperp)
    call broadcast (reality)
    call broadcast (box)
    call broadcast (nkpolar)
    call broadcast (gridopt_switch)

    allocate (akx(nakx), ikx(nakx))
    allocate (aky(naky), iky(naky))

! Set up grids
    select case (gridopt_switch)
    case (gridopt_single)
       
       aky = akperp  ; iky = 0
       akx = 0.      ; ikx = 0

    case (gridopt_box)
       do i = 1, naky
          iky(i) = i-1
          aky(i) = real(i-1)/y0
       end do
       
       do i = 1, (nakx+1)/2
          ikx(i) = i-1
          akx(i) = real(i-1)/x0
       end do
       
       do i = (nakx+1)/2+1, nakx
          ikx(i) = i-nakx-1
          akx(i) = real(i-nakx-1)/x0
       end do
    end select

  end subroutine init_kgrids

end module kgrids

