module theta_grid
  implicit none


  public :: init_theta_grid   ! initialize this module
  public :: gradpar, jacob    ! related to d/dtheta; mostly holdovers from gs2
  public :: ntheta            ! # of grid points along field line if nperiod = 1
  public :: ntgrid            ! total # of grid points along field line
  public :: nperiod           ! holdover from gs2; probably = 1 for foreseeable future
  public :: theta             ! Coordinate along the field line; -pi <= theta <= pi
  public :: delthet           ! theta(i+1)-theta(i)

! z0 sets box length along the field line: Lz = 2 * pi * z0, since box is 2*pi in extent.

  real, dimension(:), allocatable :: theta
  real :: gradpar, z0, jacob, delthet
  integer :: ntheta, ntgrid, nperiod

  private

contains

  subroutine init_theta_grid

    use mp, only: proc0, broadcast
    use file_utils, only: input_unit, input_unit_exist
    use constants, only: pi
    implicit none
    logical, save :: initialized = .false.
    integer :: in_file, i
    logical :: exist

    namelist /theta_grid/ z0, ntheta, nperiod

    if (initialized) return
    initialized = .true.

    if (proc0) then
       
       z0 = 1.0
       ntheta = 24
       nperiod = 1
       in_file = input_unit_exist("theta_grid", exist)
       if (exist) read (unit=input_unit("theta_grid"), nml=theta_grid)
       
       gradpar = 1./z0  ! Box length is 2*pi*z0 along the equilibrium magnetic field
       jacob = 1.0/gradpar

       ntgrid = ntheta/2 + (nperiod-1)*ntheta

       allocate (theta(-ntgrid:ntgrid))
       theta = (/ (real(i)*2.0*pi/real(ntheta), i=-ntgrid,ntgrid) /)

       delthet = theta(1) - theta(0)
       
    end if

    call broadcast (ntheta)
    call broadcast (ntgrid)
    call broadcast (nperiod)
    
    if (.not. proc0) then
       allocate (theta(-ntgrid:ntgrid))
    end if

    call broadcast (theta)
    call broadcast (delthet)
    call broadcast (gradpar)
    call broadcast (jacob)

  end subroutine init_theta_grid

end module theta_grid
