module convert

  private
  public :: c2r, r2c

  interface c2r
     module procedure x1c2r
     module procedure x2c2r
     module procedure x3c2r
     module procedure x4c2r
  end interface

  interface r2c
     module procedure x1r2c
     module procedure x2r2c
     module procedure x3r2c
     module procedure x4r2c
  end interface

contains 

  subroutine x4c2r(a, a_ri)

    complex, dimension(:,:,:,:) :: a
    real, dimension(:,:,:,:,:) :: a_ri

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x4c2r: size(a, 1) does not match size(a_ri, 2)')
    if(size(a, 2) /= size(a_ri, 3)) call aborter (6, 'x4c2r: size(a, 2) does not match size(a_ri, 3)')
    if(size(a, 3) /= size(a_ri, 4)) call aborter (6, 'x4c2r: size(a, 3) does not match size(a_ri, 4)')
    if(size(a, 4) /= size(a_ri, 5)) call aborter (6, 'x4c2r: size(a, 4) does not match size(a_ri, 5)')
    a_ri(1,:,:,:,:) = real(a(:,:,:,:))
    a_ri(2,:,:,:,:) = aimag(a(:,:,:,:))

  end subroutine x4c2r

  subroutine x4r2c(a, a_ri)

    real, dimension(:,:,:,:,:) :: a_ri
    complex, dimension(:,:,:,:) :: a

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x4r2c: size(a, 1) does not match size(a_ri, 2)')
    if(size(a, 2) /= size(a_ri, 3)) call aborter (6, 'x4r2c: size(a, 2) does not match size(a_ri, 3)')
    if(size(a, 3) /= size(a_ri, 4)) call aborter (6, 'x4r2c: size(a, 3) does not match size(a_ri, 4)')
    if(size(a, 4) /= size(a_ri, 5)) call aborter (6, 'x4r2c: size(a, 4) does not match size(a_ri, 5)')
    a(:,:,:,:) = cmplx(a_ri(1,:,:,:,:), a_ri(2,:,:,:,:))

  end subroutine x4r2c

  subroutine x3c2r(a, a_ri)

    complex, dimension(:,:,:) :: a
    real, dimension(:,:,:,:) :: a_ri

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x3c2r: size(a, 1) does not match size(a_ri, 2)')
    if(size(a, 2) /= size(a_ri, 3)) call aborter (6, 'x3c2r: size(a, 2) does not match size(a_ri, 3)')
    if(size(a, 3) /= size(a_ri, 4)) call aborter (6, 'x3c2r: size(a, 3) does not match size(a_ri, 4)')
    a_ri(1,:,:,:) = real(a(:,:,:))
    a_ri(2,:,:,:) = aimag(a(:,:,:))

  end subroutine x3c2r

  subroutine x3r2c(a, a_ri)

    real, dimension(:,:,:,:) :: a_ri
    complex, dimension(:,:,:) :: a

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x3r2c: size(a, 1) does not match size(a_ri, 2)')
    if(size(a, 2) /= size(a_ri, 3)) call aborter (6, 'x3r2c: size(a, 2) does not match size(a_ri, 3)')
    if(size(a, 3) /= size(a_ri, 4)) call aborter (6, 'x3r2c: size(a, 3) does not match size(a_ri, 4)')
    a(:,:,:) = cmplx(a_ri(1,:,:,:), a_ri(2,:,:,:))

  end subroutine x3r2c

  subroutine x2c2r(a, a_ri)

    complex, dimension(:,:) :: a
    real, dimension(:,:,:) :: a_ri

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x2c2r: size(a, 1) does not match size(a_ri, 2)')
    if(size(a, 2) /= size(a_ri, 3)) call aborter (6, 'x2c2r: size(a, 2) does not match size(a_ri, 3)')
    a_ri(1,:,:) = real(a(:,:))
    a_ri(2,:,:) = aimag(a(:,:))

  end subroutine x2c2r

  subroutine x2r2c(a, a_ri)

    real, dimension(:,:,:) :: a_ri
    complex, dimension(:,:) :: a

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x2r2c: size(a, 1) does not match size(a_ri, 2)')
    if(size(a, 2) /= size(a_ri, 3)) call aborter (6, 'x2r2c: size(a, 2) does not match size(a_ri, 3)')
    a(:,:) = cmplx(a_ri(1,:,:), a_ri(2,:,:))

  end subroutine x2r2c

  subroutine x1c2r(a, a_ri)

    complex, dimension(:) :: a
    real, dimension(:,:) :: a_ri

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x2c2r: size(a, 1) does not match size(a_ri, 2)')
    a_ri(1,:) = real(a(:))
    a_ri(2,:) = aimag(a(:))

  end subroutine x1c2r

  subroutine x1r2c(a, a_ri)

    real, dimension(:,:) :: a_ri
    complex, dimension(:) :: a

    if(size(a, 1) /= size(a_ri, 2)) call aborter (6, 'x2r2c: size(a, 1) does not match size(a_ri, 2)')
    a(:) = cmplx(a_ri(1,:), a_ri(2,:))

  end subroutine x1r2c

  Subroutine Aborter(iunit,ierrmsg)
!----------------------------------------------------------------------
!  ABORT A PROGRAM AFTER A FATAL ERROR CONDITION IS DETECTED.
!
!
! input: iunit  Unit Number of the file to write error messages to.
!        ierrmsg  An error message to write ilunerr
!
!       The advantage of using this subroutine is that it will
!       generate a traceback showing the chain of subroutines which
!       eventually bombed, and it forces an arithmetic error which
!       the job control system can detect as a fatal error.
!
    character ierrmsg*(*)
    common /abortcmn/ zz0,zz1
!
! zz0 is in a common block to prevent an optimizing compiler
! from evaluating 1.0/zz0 during compilation rather than during
! execution.
!
    write(iunit,1001)
1001 format(//' %ABORTER:  ** FATAL ERROR.  ABORT SUBROUTINE CALLED **'//)
  
    write(iunit,1002)ierrmsg
1002 format(1x,a,//)

! on CRAY's and VAXes:
! generate a divide-by-zero error:
    zz1=1.0/zz0         !^
    ilunerr=5           !^

! on the DecStation:
!^      call exit(1)

    write(ilunerr,1010) zz0
1010 format(' ?ABORTER-- ZZ0= ',1PE11.4,' AND STILL EXECUTING...')

  end subroutine aborter

end module convert



module agk_io

  implicit none

  private

  public :: init_agk_io, nc_final_fields, nc_final_epar
  public :: nc_final_moments, nc_finish
  public :: nc_qflux, nc_vflux, nc_pflux, nc_loop, nc_loop_moments
  public :: nc_loop_movie

  logical, parameter :: serial_io = .true.
  integer :: ncid

  integer :: naky_dim, nakx_dim, nttot_dim, negrid_dim, nlambda_dim, nspec_dim
  integer :: nsign_dim, time_dim, char10_dim, char200_dim, ri_dim, nlines_dim, nheat_dim
  
  integer, dimension (5) :: field_dim, final_mom_dim, heatk_dim
  integer, dimension (4) :: omega_dim, fluxk_dim, final_field_dim, loop_mom_dim
  integer, dimension (3) :: fluxx_dim
  integer, dimension (3) :: mode_dim, phase_dim, loop_phi_dim, heat_dim
  integer, dimension (2) :: kx_dim, ky_dim, om_dim, flux_dim, nin_dim, fmode_dim

  ! added by EAB 03/05/04 for movies
  logical :: my_make_movie
  integer :: ncid_movie
  integer :: nx_dim, ny_dim, nth_dim, time_movie_dim
  integer, dimension (4) :: xmode_dim
  integer :: nx_id, ny_id, nth_id, x_id, y_id, th_id, time_movie_id
  integer :: phi_by_xmode_id, apar_by_xmode_id, bpar_by_xmode_id
  integer :: code_id_movie

  integer :: nakx_id, naky_id, nttot_id, akx_id, aky_id, theta_id, nspec_id
  integer :: time_id, phi2_id, apar2_id, bpar2_id, nproc_id, nmesh_id
  integer :: phi2_by_mode_id, apar2_by_mode_id, bpar2_by_mode_id
  integer :: phtot_id, dmix_id, kperpnorm_id
  integer :: phi2_by_kx_id, apar2_by_kx_id, bpar2_by_kx_id
  integer :: phi2_by_ky_id, apar2_by_ky_id, bpar2_by_ky_id
  integer :: phi0_id, apar0_id, bpar0_id, sourcefac_id
  integer :: omega_id, omegaavg_id, phase_id, phiavg_id
  integer :: es_heat_flux_id, es_mom_flux_id, es_part_flux_id
  integer :: es_heat_par_id, es_heat_perp_id
  integer :: apar_heat_flux_id, apar_mom_flux_id, apar_part_flux_id
  integer :: apar_heat_par_id, apar_heat_perp_id
  integer :: bpar_heat_flux_id, bpar_mom_flux_id, bpar_part_flux_id
  integer :: bpar_heat_par_id, bpar_heat_perp_id
  integer :: hflux_tot_id, zflux_tot_id, vflux_tot_id
  integer :: es_heat_by_k_id, es_mom_by_k_id, es_part_by_k_id
  integer :: apar_heat_by_k_id, apar_mom_by_k_id, apar_part_by_k_id
  integer :: bpar_heat_by_k_id, bpar_mom_by_k_id, bpar_part_by_k_id
  integer :: phi_t_id, apar_t_id, bpar_t_id
  integer :: phi_norm_id, apar_norm_id, bpar_norm_id
  integer :: phi_id, apar_id, bpar_id, epar_id
  integer :: ntot_id, density_id, upar_id, tpar_id, tperp_id
  integer :: ntot2_id, ntot2_by_mode_id, ntot20_id, ntot20_by_mode_id
  integer :: phi00_id, ntot00_id, density00_id, upar00_id, tpar00_id, tperp00_id
  integer :: qflux_neo_by_k_id, pflux_neo_by_k_id, input_id
  integer :: charge_id, mass_id, dens_id, temp_id, tprim_id, fprim_id
  integer :: uprim_id, uprim2_id, nu_id, spec_type_id
  integer :: jacob_id, gradpar_id
  integer :: code_id, datestamp_id, timestamp_id, timezone_id
  integer :: h_energy_id, h_energy_dot_id, h_antenna_id
  integer :: h_eapar_id, h_ebpar_id
  integer :: h_delfs2_id, h_hs2_id, h_phis2_id
  integer :: h_hypervisc_id, h_hyperres_id, h_collisions_id
  integer :: h_gradients_id, h_hypercoll_id, h_heating_id
  integer :: h_imp_colls_id
  integer :: hk_energy_id, hk_energy_dot_id, hk_antenna_id
  integer :: hk_eapar_id, hk_ebpar_id
  integer :: hk_delfs2_id, hk_hs2_id, hk_phis2_id
  integer :: hk_hypervisc_id, hk_hyperres_id, hk_collisions_id
  integer :: hk_gradients_id, hk_hypercoll_id, hk_heating_id, hk_imp_colls_id

  real :: zero
  
  include 'netcdf.inc'
  
contains

  subroutine init_agk_io (write_nl_flux, write_omega, &
      write_hrate, &
      make_movie, nmovie_tot)
!David has made some changes to this subroutine (may 2005) now should be able to do movies for 
!linear box runs as well as nonlinear box runs.

    use mp, only: proc0, barrier
    use file_utils, only: run_name
    use netcdf_mod
    use agk_transforms, only: init_transforms
    use kgrids, only: naky, nakx,nx,ny
    use agk_layouts, only: yxf_lo
    use theta_grid, only: ntgrid
    use le_grids, only: nlambda, negrid
    use species, only: nspec

    logical :: write_nl_flux, write_omega, write_hrate, make_movie
    logical :: accelerated
    character (300) :: filename, filename_movie
    integer :: ierr         ! 0 if initialization is successful
    integer :: nmovie_tot
    integer :: status


    zero = epsilon(0.0)
    ierr = 0

    call netcdf_init(serial_io)

    status = nf_noerr

    filename = trim(trim(run_name)//'.out.nc')
    if(serial_io) then
       ! only proc0 opens the file:
       if (proc0) &
            status = nf_create(trim(filename), 0, ncid) 
    else
       ! all processors open the file
       call barrier
       status = nf_create(trim(filename), 0, ncid) ! overwrites old
       call barrier
    endif

    if(status /= nf_noerr) then
       write(*,*) 'Unable to create netcdf file ', filename
       write(*,*) nf_strerror(status)
       ierr = 1
    endif

    ! added by EAB 03/2004 for making movie netcdf files
    my_make_movie = make_movie
    if(my_make_movie) then
    !perhaps put a call to init_y_transform
!    call init_y_transform_layouts  
    call init_transforms(ntgrid, naky, nakx, nlambda, negrid, nspec, nx, ny, accelerated)
       filename_movie = trim(trim(run_name)//'.movie.nc')
       if(serial_io) then
          ! only proc0 opens the file:
          if (proc0) &
               status = nf_create(trim(filename_movie), 0, ncid_movie) 
       else
          ! all processors open the file
          call barrier
          status = nf_create(trim(filename_movie), 0, ncid_movie) ! overwrites old
          call barrier
       endif
       
       if(status /= nf_noerr) then
          write(*,*) 'Unable to create netcdf file for movies', filename_movie
          write(*,*) nf_strerror(status)
          ierr = 1
       endif
    endif

    if (proc0) then
       call define_dims (nmovie_tot)
       call define_vars (write_nl_flux, write_omega, write_hrate)
       call nc_grids
       call nc_species
       call nc_geo
    end if

  end subroutine init_agk_io

  subroutine define_dims (nmovie_tot)

    use file_utils, only: num_input_lines
    use kgrids, only: naky, nakx,nx,ny
    use agk_layouts, only: yxf_lo    
    use theta_grid, only: ntgrid
    use le_grids, only: nlambda, negrid
    use species, only: nspec
    use netcdf_mod

    integer, intent (in) :: nmovie_tot
    integer :: status


    status = netcdf_def_dim(ncid, 'ky', naky, naky_dim)
    status = netcdf_def_dim(ncid, 'kx', nakx, nakx_dim)
    status = netcdf_def_dim(ncid, 'theta', 2*ntgrid+1, nttot_dim)
    status = netcdf_def_dim(ncid, 'lambda', nlambda, nlambda_dim)
    status = netcdf_def_dim(ncid, 'egrid', negrid, negrid_dim)
    status = netcdf_def_dim(ncid, 'species', nspec, nspec_dim)
    status = netcdf_def_dim(ncid, 'sign', 2, nsign_dim)
    status = netcdf_def_dim(ncid, 't', NF_UNLIMITED, time_dim)
    status = netcdf_def_dim(ncid, 'char10', 10, char10_dim)
    status = netcdf_def_dim(ncid, 'char200', 200, char200_dim)
    status = netcdf_def_dim(ncid, 'nlines', num_input_lines, nlines_dim)
    status = netcdf_def_dim(ncid, 'ri', 2, ri_dim)
    status = netcdf_def_dim(ncid, 'nheat', 7, nheat_dim)

    ! added by EAB 03/05/04 for movies
    if(my_make_movie) then



       
       nx=yxf_lo%nx
       ny=yxf_lo%ny
       status = netcdf_def_dim(ncid_movie, 'x', nx, nx_dim)
       status = netcdf_def_dim(ncid_movie, 'y', ny, ny_dim)
       status = netcdf_def_dim(ncid_movie, 'theta', 2*ntgrid+1, nth_dim)
!
! can only have one NF_UNLIMITED, so:
!
!       status = netcdf_def_dim(ncid_movie, 't', NF_UNLIMITED, time_movie_dim)
       status = netcdf_def_dim(ncid_movie, 'tm', nmovie_tot, time_movie_dim)
    endif

  end subroutine define_dims

  subroutine nc_grids

    use theta_grid, only: ntgrid, theta, ntheta
    use kgrids, only: naky, nakx, akx, aky, nx, ny
    use agk_layouts, only: yxf_lo
    use species, only: nspec
    use le_grids, only: negrid, nlambda
    use nonlinear_terms, only: nonlin
    use netcdf_mod
    use constants

    integer :: status
    real :: nmesh

    real, dimension(:), allocatable :: x, y
    integer :: ik, it


    status = netcdf_put_var (ncid, nttot_id, 2*ntgrid+1)
    status = netcdf_put_var (ncid, naky_id, naky)
    status = netcdf_put_var (ncid, nakx_id, nakx)
    status = netcdf_put_var (ncid, nspec_id, nspec)

    status = netcdf_put_var (ncid, akx_id, akx)
    status = netcdf_put_var (ncid, aky_id, aky)
    status = netcdf_put_var (ncid, theta_id, theta)

    if (nonlin) then
       nmesh = (2*ntgrid+1)*2*nlambda*negrid*nx*ny*nspec
    else
       nmesh = (2*ntgrid+1)*2*nlambda*negrid*nakx*naky*nspec
    end if

    status = netcdf_put_var (ncid, nmesh_id, nmesh)

    ! added by EAB 03/05/04 for movies
    if(my_make_movie) then

       allocate(x(yxf_lo%nx))
       allocate(y(yxf_lo%ny))
       do it = 1, yxf_lo%nx 
          x(it) = 2.0*pi/akx(2)*(-0.5+ real(it-1)/real(yxf_lo%nx))
       end do
       do ik = 1, yxf_lo%ny
          y(ik) = 2.0*pi/aky(2)*(-0.5+ real(ik-1)/real(yxf_lo%ny))
       end do
       
       status = netcdf_put_var (ncid_movie, nx_id, yxf_lo%nx)
       status = netcdf_put_var (ncid_movie, ny_id, yxf_lo%ny)
       status = netcdf_put_var (ncid_movie, nth_id, 2*ntgrid+1)
       status = netcdf_put_var (ncid_movie, x_id, x)
       status = netcdf_put_var (ncid_movie, y_id, y)
       status = netcdf_put_var (ncid_movie, th_id, theta)
       deallocate(x)
       deallocate(y)
    endif

  end subroutine nc_grids

  subroutine nc_finish

    use mp, only: proc0
    integer :: status

    if (proc0) then
       call save_input
       status = nf_close (ncid)
       if(my_make_movie) then
          status = nf_close (ncid_movie)
       endif
    end if

  end subroutine nc_finish

  subroutine save_input
    
    use file_utils, only: num_input_lines, get_input_unit
    use netcdf_mod

    character(200) line
    integer, dimension (2) :: nin_start, nin_count

    integer :: status, n, unit, iostat

    nin_start(1) = 1
    nin_start(2) = 1

    nin_count(2) = 1

    call get_input_unit (unit)
    rewind (unit=unit)
    do n = 1, num_input_lines
       read (unit=unit, fmt="(a)") line
       nin_count(1) = len(trim(line))
       status = nf_put_vara_text (ncid, input_id, nin_start, nin_count, line)
       nin_start(2) = nin_start(2) + 1
    end do

  end subroutine save_input

  subroutine define_vars (write_nl_flux, write_omega, write_hrate)

    use mp, only: nproc
    use species, only: nspec
    use kgrids, only: naky, nakx
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use netcdf_mod
    logical :: write_nl_flux, write_omega, write_hrate

    character (5) :: ci
    character (20) :: datestamp, timestamp, timezone
    logical :: d_fields_per = .false.
    logical :: d_neo = .false.
    
    integer :: status

    fmode_dim(1) = nakx_dim
    fmode_dim(2) = naky_dim

    mode_dim (1) = nakx_dim
    mode_dim (2) = naky_dim
    mode_dim (3) = time_dim

    kx_dim (1) = nakx_dim
    kx_dim (2) = time_dim
    
    ky_dim (1) = naky_dim
    ky_dim (2) = time_dim
    
    om_dim (1) = ri_dim
    om_dim (2) = time_dim

    omega_dim (1) = ri_dim
    omega_dim (2) = nakx_dim
    omega_dim (3) = naky_dim
    omega_dim (4) = time_dim

    phase_dim (1) = ri_dim
    phase_dim (2) = nakx_dim
    phase_dim (3) = naky_dim
    
    nin_dim(1) = char200_dim
    nin_dim(2) = nlines_dim
    
    flux_dim (1) = nspec_dim
    flux_dim (2) = time_dim

    fluxk_dim (1) = nakx_dim
    fluxk_dim (2) = naky_dim
    fluxk_dim (3) = nspec_dim
    fluxk_dim (4) = time_dim

    fluxx_dim (1) = nakx_dim
    fluxx_dim (2) = nspec_dim
    fluxx_dim (3) = time_dim

    heat_dim (1) = nspec_dim
    heat_dim (2) = nheat_dim
    heat_dim (3) = time_dim

    heatk_dim (1) = nakx_dim
    heatk_dim (2) = naky_dim
    heatk_dim (3) = nspec_dim
    heatk_dim (4) = nheat_dim
    heatk_dim (5) = time_dim

    field_dim (1) = ri_dim
    field_dim (2) = nttot_dim
    field_dim (3) = nakx_dim
    field_dim (4) = naky_dim
    field_dim (5) = time_dim
    
    final_field_dim (1) = ri_dim
    final_field_dim (2) = nttot_dim
    final_field_dim (3) = nakx_dim
    final_field_dim (4) = naky_dim

    final_mom_dim (1) = ri_dim
    final_mom_dim (2) = nttot_dim
    final_mom_dim (3) = nakx_dim
    final_mom_dim (4) = naky_dim
    final_mom_dim (5) = nspec_dim

    loop_mom_dim (1) = ri_dim
    loop_mom_dim (2) = nakx_dim
    loop_mom_dim (3) = nspec_dim
    loop_mom_dim (4) = time_dim

    loop_phi_dim (1) = ri_dim
    loop_phi_dim (2) = nakx_dim
    loop_phi_dim (3) = time_dim

    status = netcdf_put_att (ncid, nf_global, 'title', 'AstroGK Simulation Data')

    datestamp(:) = ' '
    timestamp(:) = ' '
    timezone(:) = ' '
    call date_and_time (datestamp, timestamp, timezone)
    
    status = netcdf_def_var (ncid, 'code_info', nf_char, 1, char10_dim, code_id)
    status = netcdf_put_att (ncid, code_id, 'long_name', 'AstroGK')

    ci = 'c1'
    status = netcdf_put_att (ncid, code_id, trim(ci), 'Date: '//trim(datestamp))

    ci = 'c2'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'Time: '//trim(timestamp)//' '//trim(timezone))

    ci = 'c3'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'netCDF version '//trim(NF_INQ_LIBVERS()))

    ci = 'c4'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'Units are determined with respect to reference temperature (T_ref),')

    ci = 'c5'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'reference charge (q_ref), reference mass (mass_ref),')

    ci = 'c6'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'reference field (B_ref), and reference length (a_ref)')

    ci = 'c7'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'from which one may construct rho_ref and vt_ref/a,')

    ci = 'c8'
    status = netcdf_put_att (ncid, code_id, trim(ci), &
         'which are the basic units of perpendicular length and time.')

! TT> commented out obsolete descriptions
!    ci = 'c9'
!    status = netcdf_put_att (ncid, code_id, trim(ci), &
!         'Macroscopic lengths are normalized to the minor radius.')
!
!    ci = 'c10'
!    status = netcdf_put_att (ncid, code_id, trim(ci), &
!         'The difference between rho (normalized minor radius) and rho (gyroradius)')
!
!    ci = 'c11'
!    status = netcdf_put_att (ncid, code_id, trim(ci), &
!         'should be clear from the context in which they appear below.')
! <TT

    status = netcdf_def_var (ncid, 'nproc', nf_int, 0, 0, nproc_id)
    status = netcdf_put_att (ncid, nproc_id, 'long_name', 'Number of processors')

    status = netcdf_def_var (ncid, 'nmesh', nf_double, 0, 0, nmesh_id)
    status = netcdf_put_att (ncid, nmesh_id, 'long_name', 'Number of meshpoints')

    status = netcdf_def_var (ncid, 'nkx', nf_int, 0, 0, nakx_id)
    status = netcdf_def_var (ncid, 'nky', nf_int, 0, 0, naky_id)
    status = netcdf_def_var (ncid, 'ntheta_tot', nf_int, 0, 0, nttot_id)
    status = netcdf_def_var (ncid, 'nspecies', nf_int, 0, 0, nspec_id)

    status = netcdf_def_var (ncid, 't', nf_double, 1, time_dim, time_id)
    status = netcdf_put_att (ncid, time_id, 'long_name', 'Time')
    status = netcdf_put_att (ncid, time_id, 'units', 'L/vt')

    status = netcdf_def_var (ncid, 'charge', nf_int, 1, nspec_dim, charge_id)
    status = netcdf_put_att (ncid, charge_id, 'long_name', 'Charge')
    status = netcdf_put_att (ncid, charge_id, 'units', 'q')

    status = netcdf_def_var (ncid, 'mass', nf_double, 1, nspec_dim, mass_id)
    status = netcdf_put_att (ncid, mass_id, 'long_name', 'Atomic mass')
    status = netcdf_put_att (ncid, mass_id, 'units', 'm')

    status = netcdf_def_var (ncid, 'dens', nf_double, 1, nspec_dim, dens_id)
    status = netcdf_put_att (ncid, dens_id, 'long_name', 'Density')
    status = netcdf_put_att (ncid, dens_id, 'units', 'n_e')

    status = netcdf_def_var (ncid, 'temp', nf_double, 1, nspec_dim, temp_id)
    status = netcdf_put_att (ncid, temp_id, 'long_name', 'Temperature')
    status = netcdf_put_att (ncid, temp_id, 'units', 'T')

    status = netcdf_def_var (ncid, 'tprim', nf_double, 1, nspec_dim, tprim_id)
    status = netcdf_put_att (ncid, tprim_id, 'long_name', '-1/rho dT/drho')

    status = netcdf_def_var (ncid, 'fprim', nf_double, 1, nspec_dim, fprim_id) 
    status = netcdf_put_att (ncid, fprim_id, 'long_name', '-1/rho dn/drho')

    status = netcdf_def_var (ncid, 'uprim', nf_double, 1, nspec_dim, uprim_id)
    status = netcdf_put_att (ncid, uprim_id, 'long_name', '-1/v_t du_par/drho')

    status = netcdf_def_var (ncid, 'uprim2', nf_double, 1, nspec_dim, uprim2_id)

    status = netcdf_def_var (ncid, 'nu', nf_double, 1, nspec_dim, nu_id)
    status = netcdf_put_att (ncid, nu_id, 'long_name', 'Collisionality')
    status = netcdf_put_att (ncid, nu_id, 'units', 'v_t/L')
    
    status = netcdf_def_var (ncid, 'type_of_species', nf_int, 1, nspec_dim, spec_type_id)

    status = netcdf_def_var (ncid, 'kx', nf_double, 1, nakx_dim, akx_id)
    status = netcdf_put_att (ncid, akx_id, 'long_name', 'kx rho')

    status = netcdf_def_var (ncid, 'ky', nf_double, 1, naky_dim, aky_id)
    status = netcdf_put_att (ncid, aky_id, 'long_name', 'ky rho')

    status = netcdf_def_var (ncid, 'theta', nf_double, 1, nttot_dim, theta_id)

    status = netcdf_def_var (ncid, 'gradpar', nf_double, 0, 0, gradpar_id)
    status = netcdf_def_var (ncid, 'jacob', nf_double, 0, 0, jacob_id)

    if (use_Phi) then
       status = netcdf_def_var (ncid, 'phi2', nf_double, 1, time_dim, phi2_id)
       status = netcdf_put_att (ncid, phi2_id, 'long_name', '|Potential**2|')
       status = netcdf_put_att (ncid, phi2_id, 'units', '(T/q rho/L)**2')

       status = netcdf_def_var (ncid, 'phi2_by_mode', nf_double, 3, mode_dim, phi2_by_mode_id)
       if (nakx > 1) &
       status = netcdf_def_var (ncid, 'phi2_by_kx',   nf_double, 2, kx_dim, phi2_by_kx_id)
       if (naky > 1) &
       status = netcdf_def_var (ncid, 'phi2_by_ky',   nf_double, 2, ky_dim, phi2_by_ky_id)
       status = netcdf_def_var (ncid, 'phi0',         nf_double, 4, omega_dim, phi0_id)
       if (write_nl_flux) then
          status = netcdf_def_var (ncid, 'es_heat_par',  nf_double, 2, flux_dim, es_heat_par_id)
          status = netcdf_def_var (ncid, 'es_heat_perp', nf_double, 2, flux_dim, es_heat_perp_id)
          status = netcdf_def_var (ncid, 'es_heat_flux', nf_double, 2, flux_dim, es_heat_flux_id)
          status = netcdf_def_var (ncid, 'es_mom_flux',  nf_double, 2, flux_dim, es_mom_flux_id)
          status = netcdf_def_var (ncid, 'es_part_flux', nf_double, 2, flux_dim, es_part_flux_id)
          status = netcdf_def_var (ncid, 'es_heat_by_k', nf_double, 4, fluxk_dim, es_heat_by_k_id)
          status = netcdf_def_var (ncid, 'es_mom_by_k',  nf_double, 4, fluxk_dim, es_mom_by_k_id)
          status = netcdf_def_var (ncid, 'es_part_by_k', nf_double, 4, fluxk_dim, es_part_by_k_id)
       end if
       status = netcdf_def_var (ncid, 'phi',          nf_double, 4, final_field_dim, phi_id)
       status = netcdf_put_att (ncid, phi_id, 'long_name', 'Electrostatic Potential')
       status = netcdf_put_att (ncid, phi_id, 'idl_name', '!7U!6')
       status = netcdf_put_att (ncid, phi_id, 'units', 'T/q rho/L')
       if (d_fields_per) &
            status = netcdf_def_var (ncid, 'phi_t',        nf_double, 5, field_dim, phi_t_id)
    end if

    if (use_Apar) then
       status = netcdf_def_var (ncid, 'apar2',          nf_double, 1, time_dim, apar2_id)
       status = netcdf_def_var (ncid, 'apar2_by_mode',  nf_double, 3, mode_dim, apar2_by_mode_id)
       if (nakx > 1) &
       status = netcdf_def_var (ncid, 'apar2_by_kx',    nf_double, 2, kx_dim, apar2_by_kx_id)
       if (naky > 1) &
       status = netcdf_def_var (ncid, 'apar2_by_ky',    nf_double, 2, ky_dim, apar2_by_ky_id)
       status = netcdf_def_var (ncid, 'apar0',          nf_double, 4, omega_dim, apar0_id)
       if (write_nl_flux) then
          status = netcdf_def_var (ncid, 'apar_heat_flux', nf_double, 2, flux_dim, apar_heat_flux_id)
          status = netcdf_def_var (ncid, 'apar_heat_par',  nf_double, 2, flux_dim, apar_heat_par_id)
          status = netcdf_def_var (ncid, 'apar_heat_perp', nf_double, 2, flux_dim, apar_heat_perp_id)
          status = netcdf_def_var (ncid, 'apar_mom_flux',  nf_double, 2, flux_dim, apar_mom_flux_id)
          status = netcdf_def_var (ncid, 'apar_part_flux', nf_double, 2, flux_dim, apar_part_flux_id)
          status = netcdf_def_var (ncid, 'apar_heat_by_k', nf_double, 4, fluxk_dim, apar_heat_by_k_id)
          status = netcdf_def_var (ncid, 'apar_mom_by_k',  nf_double, 4, fluxk_dim, apar_mom_by_k_id)
          status = netcdf_def_var (ncid, 'apar_part_by_k', nf_double, 4, fluxk_dim, apar_part_by_k_id)
       end if
       status = netcdf_def_var (ncid, 'apar',           nf_double, 4, final_field_dim, apar_id)
       if (d_fields_per) &
            status = netcdf_def_var (ncid, 'apar_t',  nf_double, 5, field_dim, apar_t_id)
       status = netcdf_put_att (ncid, apar2_by_mode_id, 'long_name', 'Apar squared')
       status = netcdf_put_att (ncid, apar_id, 'long_name', 'Parallel Magnetic Potential')      
       status = netcdf_put_att (ncid, apar_id, 'idl_name', '!6A!9!D#!N!6')      
       status = netcdf_put_att (ncid, apar2_id, 'long_name', 'Total A_par squared')
    end if

    if (use_Bpar) then
       status = netcdf_def_var (ncid, 'bpar2',          nf_double, 1, time_dim, bpar2_id)
       status = netcdf_def_var (ncid, 'bpar2_by_mode',  nf_double, 3, mode_dim, bpar2_by_mode_id)
       if (nakx > 1) &
       status = netcdf_def_var (ncid, 'bpar2_by_kx',    nf_double, 2, kx_dim, bpar2_by_kx_id)
       if (naky > 1) &
       status = netcdf_def_var (ncid, 'bpar2_by_ky',    nf_double, 2, ky_dim, bpar2_by_ky_id)
       status = netcdf_def_var (ncid, 'bpar0',          nf_double, 4, omega_dim, bpar0_id)
       if (write_nl_flux) then
          status = netcdf_def_var (ncid, 'bpar_heat_flux', nf_double, 2, flux_dim, bpar_heat_flux_id)
          status = netcdf_def_var (ncid, 'bpar_heat_par',  nf_double, 2, flux_dim, bpar_heat_par_id)
          status = netcdf_def_var (ncid, 'bpar_heat_perp', nf_double, 2, flux_dim, bpar_heat_perp_id)
          status = netcdf_def_var (ncid, 'bpar_mom_flux',  nf_double, 2, flux_dim, bpar_mom_flux_id)
          status = netcdf_def_var (ncid, 'bpar_part_flux', nf_double, 2, flux_dim, bpar_part_flux_id)
          status = netcdf_def_var (ncid, 'bpar_heat_by_k', nf_double, 4, fluxk_dim, bpar_heat_by_k_id)
          status = netcdf_def_var (ncid, 'bpar_mom_by_k',  nf_double, 4, fluxk_dim, bpar_mom_by_k_id)
          status = netcdf_def_var (ncid, 'bpar_part_by_k', nf_double, 4, fluxk_dim, bpar_part_by_k_id)
       end if
       status = netcdf_def_var (ncid, 'bpar',           nf_double, 4, final_field_dim, bpar_id)
       if (d_fields_per) &
            status = netcdf_def_var (ncid, 'bpar_t', nf_double, 5, field_dim, bpar_t_id)
       status = netcdf_put_att (ncid, bpar2_by_mode_id, 'long_name', 'A_perp squared')
       status = netcdf_put_att (ncid, bpar_id, 'long_name', 'delta B Parallel')      
       status = netcdf_put_att (ncid, bpar_id, 'idl_name', '!6B!9!D#!N!6')          
       status = netcdf_put_att (ncid, bpar2_id, 'long_name', 'Total A_perp squared')
    end if

    status = netcdf_def_var (ncid, 'phase', nf_double, 3, phase_dim, phase_id)
    status = netcdf_put_att (ncid, phase_id, 'long_name', 'Normalizing phase')

!    status = netcdf_def_var (ncid, 'phtot', nf_double, 3, mode_dim, phtot_id)
!    status = netcdf_def_var (ncid, 'dmix',  nf_double, 3, mode_dim, dmix_id)
!    status = netcdf_def_var (ncid, 'kperpnorm', nf_double, 3, mode_dim, kperpnorm_id)

    if (write_omega) then
       status = netcdf_def_var (ncid, 'omega',   nf_double, 4, omega_dim, omega_id)
       status = netcdf_def_var (ncid, 'omegaavg',   nf_double, 4, omega_dim, omegaavg_id)
    end if

    if (write_nl_flux) then
       status = netcdf_def_var (ncid, 'hflux_tot', nf_double, 1, time_dim, hflux_tot_id)
       status = netcdf_def_var (ncid, 'vflux_tot', nf_double, 1, time_dim, vflux_tot_id)
       status = netcdf_def_var (ncid, 'zflux_tot', nf_double, 1, time_dim, zflux_tot_id)
    end if

    if (d_neo) then
       status = netcdf_def_var (ncid, 'qflux_neo_by_k', nf_double, 4, fluxk_dim, qflux_neo_by_k_id)
       status = netcdf_def_var (ncid, 'pflux_neo_by_k', nf_double, 4, fluxk_dim, pflux_neo_by_k_id)
       status = netcdf_def_var (ncid, 'sourcefac', nf_double, 2, om_dim, sourcefac_id)
    end if

    status = netcdf_def_var (ncid, 'epar',  nf_double, 4, final_field_dim, epar_id)

!    <phi>

    status = netcdf_def_var (ncid, 'ntot2',         nf_double, 2, flux_dim,  ntot2_id)
    status = netcdf_def_var (ncid, 'ntot2_by_mode', nf_double, 4, fluxk_dim, ntot2_by_mode_id)

    status = netcdf_def_var (ncid, 'ntot20',         nf_double, 2, flux_dim,  ntot20_id)
    status = netcdf_put_att (ncid, ntot20_id, 'long_name', 'Density**2 at theta=0')
    status = netcdf_def_var (ncid, 'ntot20_by_mode', nf_double, 4, fluxk_dim, ntot20_by_mode_id)

    status = netcdf_def_var (ncid, 'ntot',    nf_double, 5, final_mom_dim, ntot_id)
    status = netcdf_def_var (ncid, 'density', nf_double, 5, final_mom_dim, density_id)
    status = netcdf_def_var (ncid, 'upar',    nf_double, 5, final_mom_dim, upar_id)
    status = netcdf_def_var (ncid, 'tpar',    nf_double, 5, final_mom_dim, tpar_id)
    status = netcdf_def_var (ncid, 'tperp',   nf_double, 5, final_mom_dim, tperp_id)

    status = netcdf_def_var (ncid, 'phi00',     nf_double, 3, loop_phi_dim, phi00_id)
    status = netcdf_def_var (ncid, 'ntot00',    nf_double, 4, loop_mom_dim, ntot00_id)
    status = netcdf_def_var (ncid, 'density00', nf_double, 4, loop_mom_dim, density00_id)
    status = netcdf_def_var (ncid, 'upar00',    nf_double, 4, loop_mom_dim, upar00_id)
    status = netcdf_def_var (ncid, 'tpar00',    nf_double, 4, loop_mom_dim, tpar00_id)
    status = netcdf_def_var (ncid, 'tperp00',   nf_double, 4, loop_mom_dim, tperp00_id)
    
    if (write_hrate) then
       status = netcdf_def_var (ncid, 'h_energy',     nf_double, 1, time_dim, h_energy_id)
       status = netcdf_def_var (ncid, 'h_energy_dot', nf_double, 1, time_dim, h_energy_dot_id)
       status = netcdf_def_var (ncid, 'h_antenna',    nf_double, 1, time_dim, h_antenna_id)
       status = netcdf_def_var (ncid, 'h_eapar',      nf_double, 1, time_dim, h_eapar_id)
       status = netcdf_def_var (ncid, 'h_ebpar',      nf_double, 1, time_dim, h_ebpar_id)

       status = netcdf_def_var (ncid, 'h_delfs2',     nf_double, 2, flux_dim, h_delfs2_id)
       status = netcdf_def_var (ncid, 'h_hs2',        nf_double, 2, flux_dim, h_hs2_id)
       status = netcdf_def_var (ncid, 'h_phis2',      nf_double, 2, flux_dim, h_phis2_id)
       status = netcdf_def_var (ncid, 'h_hypervisc',  nf_double, 2, flux_dim, h_hypervisc_id)
       status = netcdf_def_var (ncid, 'h_hyperres',   nf_double, 2, flux_dim, h_hyperres_id)
       status = netcdf_def_var (ncid, 'h_collisions', nf_double, 2, flux_dim, h_collisions_id)
       status = netcdf_def_var (ncid, 'h_gradients',  nf_double, 2, flux_dim, h_gradients_id)
       status = netcdf_def_var (ncid, 'h_hypercoll',  nf_double, 2, flux_dim, h_hypercoll_id)
       status = netcdf_def_var (ncid, 'h_imp_colls',  nf_double, 2, flux_dim, h_imp_colls_id)
       status = netcdf_def_var (ncid, 'h_heating',    nf_double, 2, flux_dim, h_heating_id)
       status = netcdf_put_att (ncid, h_heating_id, 'long_name', 'Total heating by species')

       status = netcdf_def_var (ncid, 'hk_energy',    nf_double, 3, mode_dim, hk_energy_id)
       status = netcdf_def_var (ncid, 'hk_energy_dot',nf_double, 3, mode_dim, hk_energy_dot_id)
       status = netcdf_def_var (ncid, 'hk_antenna',   nf_double, 3, mode_dim, hk_antenna_id)
       status = netcdf_def_var (ncid, 'hk_eapar',     nf_double, 3, mode_dim, hk_eapar_id)
       status = netcdf_def_var (ncid, 'hk_ebpar',     nf_double, 3, mode_dim, hk_ebpar_id)

       status = netcdf_def_var (ncid, 'hk_delfs2',     nf_double, 4, fluxk_dim, hk_delfs2_id)
       status = netcdf_def_var (ncid, 'hk_hs2',        nf_double, 4, fluxk_dim, hk_hs2_id)
       status = netcdf_def_var (ncid, 'hk_phis2',      nf_double, 4, fluxk_dim, hk_phis2_id)
       status = netcdf_def_var (ncid, 'hk_hypervisc',  nf_double, 4, fluxk_dim, hk_hypervisc_id)
       status = netcdf_def_var (ncid, 'hk_hyperres',   nf_double, 4, fluxk_dim, hk_hyperres_id)
       status = netcdf_def_var (ncid, 'hk_collisions', nf_double, 4, fluxk_dim, hk_collisions_id)
       status = netcdf_def_var (ncid, 'hk_gradients',  nf_double, 4, fluxk_dim, hk_gradients_id)
       status = netcdf_def_var (ncid, 'hk_imp_colls',  nf_double, 4, fluxk_dim, hk_imp_colls_id)
       status = netcdf_def_var (ncid, 'hk_hypercoll',  nf_double, 4, fluxk_dim, hk_hypercoll_id)
       status = netcdf_def_var (ncid, 'hk_heating',    nf_double, 4, fluxk_dim, hk_heating_id)
       status = netcdf_put_att (ncid, hk_heating_id, 'long_name', 'Total heating by species and mode')
    end if

!    status = netcdf_put_att (ncid, phtot_id, 'long_name', 'Field amplitude')

    status = netcdf_def_var (ncid, 'input_file', nf_char, 2, nin_dim, input_id)
    status = netcdf_put_att (ncid, input_id, 'long_name', 'Input file')

    status = nf_enddef (ncid)  ! out of definition mode

    status = netcdf_put_var (ncid, nproc_id, nproc)

        ! added by EAB 03/05/04 for movies
    if(my_make_movie) then
       xmode_dim (1) = nx_dim
       xmode_dim (2) = ny_dim
       xmode_dim (3) = nth_dim
       xmode_dim (4) = time_movie_dim
       status = netcdf_put_att (ncid_movie, nf_global, 'title', 'AstroGK Simulation x,y,theta Data for Movies')
       
       status = netcdf_def_var (ncid_movie, 'nx', nf_int, 0, 0, nx_id)
       status = netcdf_def_var (ncid_movie, 'ny', nf_int, 0, 0, ny_id)
       status = netcdf_def_var (ncid_movie, 'ntheta', nf_int, 0, 0, nth_id)    
       status = netcdf_def_var (ncid_movie, 'tm', nf_double, 1, time_movie_dim, time_movie_id)
       status = netcdf_put_att (ncid_movie, time_movie_id, 'long_name', 'Time')
       status = netcdf_put_att (ncid_movie, time_movie_id, 'units', 'L/vt')
       status = netcdf_def_var (ncid_movie, 'x', nf_double, 1, nx_dim, x_id)
       status = netcdf_put_att (ncid_movie, x_id, 'long_name', 'x / rho')
       status = netcdf_def_var (ncid_movie, 'y', nf_double, 1, ny_dim, y_id)
       status = netcdf_put_att (ncid_movie, y_id, 'long_name', 'y / rho')
       status = netcdf_def_var (ncid_movie, 'theta', nf_double, 1, nth_dim, th_id)
       if(use_Phi) then
          status = netcdf_def_var (ncid_movie, 'phi_by_xmode', nf_double, 4, xmode_dim, phi_by_xmode_id)
       end if
! TEMPORARY BD
       status = netcdf_def_var (ncid_movie, 'density_by_xmod', nf_double, 4, xmode_dim, apar_by_xmode_id)
!       if(use_Apar) then
!          status = netcdf_def_var (ncid_movie, 'apar_by_xmode', nf_double, 4, xmode_dim, apar_by_xmode_id)
!       end if
       if(use_Bpar) then
          status = netcdf_def_var (ncid_movie, 'bpar_by_xmode', nf_double, 4, xmode_dim, bpar_by_xmode_id)
       end if
       status = nf_enddef (ncid_movie)  ! out of definition mode
    endif

  end subroutine define_vars

  subroutine nc_final_fields

    use netcdf_mod, only: netcdf_put_var
    use convert, only: c2r
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use fields_arrays, only: phi, apar, bpar
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    real, dimension (2, 2*ntgrid+1, nakx, naky) :: ri3
    integer :: status

    if (use_Phi) then
       call c2r (phi, ri3)
       status = netcdf_put_var(ncid, phi_id, ri3)
    end if

    if (use_Apar) then
       call c2r (apar, ri3)
       status = netcdf_put_var(ncid, apar_id, ri3)
    end if

    if (use_Bpar) then
       call c2r (bpar, ri3)
       status = netcdf_put_var(ncid, bpar_id, ri3)
    end if

  end subroutine nc_final_fields

  subroutine nc_final_epar (epar)

    use netcdf_mod, only: netcdf_put_var
    use convert, only: c2r
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    complex, dimension (:,:,:), intent (in) :: epar
    real, dimension (2, 2*ntgrid+1, nakx, naky) :: ri3
    integer :: status

    call c2r (epar, ri3)
    status = netcdf_put_var(ncid, epar_id, ri3)

  end subroutine nc_final_epar

  subroutine nc_final_moments (ntot, density, upar, tpar, tperp)

    use netcdf_mod, only: netcdf_put_var
    use convert, only: c2r

    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    use species, only: nspec

    complex, dimension (:,:,:,:), intent (in) :: ntot, density, upar, tpar, tperp
    real, dimension (2, 2*ntgrid+1, nakx, naky, nspec) :: ri4
    integer :: status

    call c2r (ntot, ri4)
    status = netcdf_put_var(ncid, ntot_id, ri4)
    
    call c2r (density, ri4)
    status = netcdf_put_var(ncid, density_id, ri4)
    
    call c2r (upar, ri4)
    status = netcdf_put_var(ncid, upar_id, ri4)
    
    call c2r (tpar, ri4)
    status = netcdf_put_var(ncid, tpar_id, ri4)

    call c2r (tperp, ri4)
    status = netcdf_put_var(ncid, tperp_id, ri4)

  end subroutine nc_final_moments

  subroutine nc_loop_moments (nout, ntot2, ntot2_by_mode, ntot20, ntot20_by_mode, &
       phi00, ntot00, density00, upar00, tpar00, tperp00)

    use netcdf_mod, only: netcdf_put_vara
    use convert, only: c2r

    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    use species, only: nspec

    integer, intent (in) :: nout
    real, dimension (:), intent (in) :: ntot2, ntot20
    real, dimension (:,:,:), intent (in) :: ntot2_by_mode, ntot20_by_mode
    complex, dimension (:), intent (in) :: phi00
    complex, dimension (:,:), intent (in) :: ntot00, density00, upar00, tpar00, tperp00
    real, dimension (2, nakx, nspec) :: ri2
    real, dimension (2, nakx) :: ri1
    integer, dimension (2) :: start, count
    integer, dimension (3) :: start3, count3
    integer, dimension (4) :: start00, count00, start4, count4
    integer :: status

    start00(1) = 1
    start00(2) = 1
    start00(3) = 1
    start00(4) = nout
    
    count00(1) = 2
    count00(2) = nakx
    count00(3) = nspec
    count00(4) = 1

    start3(1) = 1
    start3(2) = 1
    start3(3) = nout
    
    count3(1) = 2
    count3(2) = nakx
    count3(3) = 1

    start(1) = 1
    start(2) = nout
    
    count(1) = nspec
    count(2) = 1

    start4(1) = 1
    start4(2) = 1
    start4(3) = 1
    start4(4) = nout
    
    count4(1) = nakx
    count4(2) = naky
    count4(3) = nspec
    count4(4) = 1

    status = netcdf_put_vara(ncid, ntot2_id, start, count, ntot2)
    status = netcdf_put_vara(ncid, ntot2_by_mode_id, start4, count4, ntot2_by_mode)

    status = netcdf_put_vara(ncid, ntot20_id, start, count, ntot20)
    status = netcdf_put_vara(ncid, ntot20_by_mode_id, start4, count4, ntot20_by_mode)

    call c2r (phi00, ri1)
    status = netcdf_put_vara(ncid, phi00_id, start3, count3, ri1)
    
    call c2r (ntot00, ri2)
    status = netcdf_put_vara(ncid, ntot00_id, start00, count00, ri2)
    
    call c2r (density00, ri2)
    status = netcdf_put_vara(ncid, density00_id, start00, count00, ri2)
    
    call c2r (upar00, ri2)
    status = netcdf_put_vara(ncid, upar00_id, start00, count00, ri2)
    
    call c2r (tpar00, ri2)
    status = netcdf_put_vara(ncid, tpar00_id, start00, count00, ri2)

    call c2r (tperp00, ri2)
    status = netcdf_put_vara(ncid, tperp00_id, start00, count00, ri2)

  end subroutine nc_loop_moments

  subroutine nc_qflux (nout, qheat, qmheat, qbheat, &
       heat_par,  mheat_par,  bheat_par, &
       heat_perp, mheat_perp, bheat_perp, &
       heat_fluxes, mheat_fluxes, bheat_fluxes, hflux_tot)

    use species, only: nspec
    use kgrids, only: naky, nakx
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use netcdf_mod, only: netcdf_put_vara, netcdf_put_var1

    integer, intent (in) :: nout
    real, dimension (:,:,:), intent (in) :: qheat, qmheat, qbheat
    real, dimension (:), intent (in) :: heat_par, mheat_par, bheat_par
    real, dimension (:), intent (in) :: heat_perp, mheat_perp, bheat_perp
    real, dimension (:), intent (in) :: heat_fluxes, mheat_fluxes, bheat_fluxes
    real, intent (in) :: hflux_tot
    integer, dimension (2) :: start, count
    integer, dimension (4) :: start4, count4
    integer :: status

    start4(1) = 1
    start4(2) = 1
    start4(3) = 1
    start4(4) = nout
    
    count4(1) = nakx
    count4(2) = naky
    count4(3) = nspec
    count4(4) = 1

    start(1) = 1
    start(2) = nout
    
    count(1) = nspec
    count(2) = 1

    if (use_Phi) then
       status = netcdf_put_vara(ncid, es_heat_flux_id, start, count, heat_fluxes)
       status = netcdf_put_vara(ncid, es_heat_par_id,  start, count, heat_par)
       status = netcdf_put_vara(ncid, es_heat_perp_id, start, count, heat_perp)
       status = netcdf_put_vara(ncid, es_heat_by_k_id, start4, count4, qheat)
    end if

    if (use_Apar) then
       status = netcdf_put_vara(ncid, apar_heat_flux_id, start, count, mheat_fluxes)
       status = netcdf_put_vara(ncid, apar_heat_par_id,  start, count, mheat_par)
       status = netcdf_put_vara(ncid, apar_heat_perp_id, start, count, mheat_perp)
       status = netcdf_put_vara(ncid, apar_heat_by_k_id, start4, count4, qmheat)
    end if

    if (use_Bpar) then
       status = netcdf_put_vara(ncid, bpar_heat_flux_id, start, count, bheat_fluxes)
       status = netcdf_put_vara(ncid, bpar_heat_par_id,  start, count, bheat_par)
       status = netcdf_put_vara(ncid, bpar_heat_perp_id, start, count, bheat_perp)
       status = netcdf_put_vara(ncid, bpar_heat_by_k_id, start4, count4, qbheat)
    end if
    
    status = netcdf_put_var1(ncid, hflux_tot_id, nout, hflux_tot)

  end subroutine nc_qflux

  subroutine nc_pflux (nout, pflux, pmflux, pbflux, &
       part_fluxes, mpart_fluxes, bpart_fluxes, zflux_tot)

    use species, only: nspec
    use kgrids, only: naky, nakx
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use netcdf_mod, only: netcdf_put_vara, netcdf_put_var1

    integer, intent (in) :: nout
    real, dimension (:,:,:), intent (in) :: pflux, pmflux, pbflux
    real, dimension(:), intent (in) :: part_fluxes, mpart_fluxes, bpart_fluxes
    real, intent (in) :: zflux_tot
    integer, dimension (2) :: start, count
    integer, dimension (4) :: start4, count4
    integer :: status

    start4(1) = 1
    start4(2) = 1
    start4(3) = 1
    start4(4) = nout
    
    count4(1) = nakx
    count4(2) = naky
    count4(3) = nspec
    count4(4) = 1

    start(1) = 1
    start(2) = nout
    
    count(1) = nspec
    count(2) = 1

    if (use_Phi) then
       status = netcdf_put_vara(ncid, es_part_flux_id, start, count, part_fluxes)
       status = netcdf_put_vara(ncid, es_part_by_k_id, start4, count4, pflux)
    end if

    if (use_Apar) then
       status = netcdf_put_vara(ncid, apar_part_flux_id, start, count, mpart_fluxes)
       status = netcdf_put_vara(ncid, apar_part_by_k_id, start4, count4, pmflux)
    end if

    if (use_Bpar) then
       status = netcdf_put_vara(ncid, bpar_part_flux_id, start, count, bpart_fluxes)
       status = netcdf_put_vara(ncid, bpar_part_by_k_id, start4, count4, pbflux)
    end if
    
    status = netcdf_put_var1(ncid, zflux_tot_id, nout, zflux_tot)

  end subroutine nc_pflux

  subroutine nc_vflux (nout, vflux, vmflux, vbflux, &
       mom_fluxes, mmom_fluxes, bmom_fluxes, vflux_tot)

    use species, only: nspec
    use kgrids, only: naky, nakx
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use netcdf_mod, only: netcdf_put_vara, netcdf_put_var1

    integer, intent (in) :: nout
    real, dimension (:,:,:), intent (in) :: vflux, vmflux, vbflux
    real, dimension(:), intent (in) :: mom_fluxes, mmom_fluxes, bmom_fluxes
    real, intent (in) :: vflux_tot
    integer, dimension (2) :: start, count
    integer, dimension (4) :: start4, count4
    integer :: status

    start4(1) = 1
    start4(2) = 1
    start4(3) = 1
    start4(4) = nout
    
    count4(1) = nakx
    count4(2) = naky
    count4(3) = nspec
    count4(4) = 1

    start(1) = 1
    start(2) = nout
    
    count(1) = nspec
    count(2) = 1

    if (use_Phi) then
       status = netcdf_put_vara(ncid, es_mom_flux_id, start, count, mom_fluxes)
       status = netcdf_put_vara(ncid, es_mom_by_k_id, start4, count4, vflux)
    end if

    if (use_Apar) then
       status = netcdf_put_vara(ncid, apar_mom_flux_id, start, count, mmom_fluxes)
       status = netcdf_put_vara(ncid, apar_mom_by_k_id, start4, count4, vmflux)
    end if

    if (use_Bpar) then
       status = netcdf_put_vara(ncid, bpar_mom_flux_id, start, count, bmom_fluxes)
       status = netcdf_put_vara(ncid, bpar_mom_by_k_id, start4, count4, vbflux)
    end if
    
    status = netcdf_put_var1(ncid, vflux_tot_id, nout, vflux_tot)

  end subroutine nc_vflux

  subroutine nc_loop (nout, time, fluxfac, &
       phi0,   phi2,   phi2_by_mode, &! phiavg, &
       apar0,  apar2,  apar2_by_mode, &
       bpar0, bpar2, bpar2_by_mode, &
       h, hk, omega, omegaavg, phitot, write_omega, write_hrate)

    use agk_heating, only: heating_diagnostics, hk_repack
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use netcdf_mod, only: netcdf_put_var1, netcdf_put_vara
    use kgrids, only: naky, nakx
    use species, only: nspec
    use convert, only: c2r

    integer, intent (in) :: nout
    real, intent (in) :: time, phi2, apar2, bpar2
    real, dimension (:), intent (in) :: fluxfac
    complex, dimension(:,:), intent (in) :: phi0, apar0, bpar0, omega, omegaavg !, phiavg
    real, dimension(:,:), intent (in) :: phi2_by_mode, apar2_by_mode, bpar2_by_mode, phitot
    type(heating_diagnostics), intent (in) :: h
    type(heating_diagnostics), dimension(:,:), intent (in) :: hk
    logical :: write_omega, write_hrate
    real, dimension (nakx) :: field2_by_kx
    real, dimension (naky) :: field2_by_ky
    real, dimension (2, nakx, naky) :: ri2
    real, dimension (nakx, naky, nspec) :: tmps
    complex, dimension (nakx, naky) :: tmp
    integer, dimension (4) :: start0, count0, start4, count4
    integer, dimension (3) :: start, count, starth, counth
    integer, dimension (2) :: startx, countx, starty, county, starts, counts
    integer :: status, it, ik, is

    status = netcdf_put_var1(ncid, time_id, nout, time)

    start(1) = 1
    start(2) = 1
    start(3) = nout

    count(1) = nakx
    count(2) = naky
    count(3) = 1

    start0(1) = 1
    start0(2) = 1
    start0(3) = 1
    start0(4) = nout

    count0(1) = 2
    count0(2) = nakx
    count0(3) = naky
    count0(4) = 1

    start4(1) = 1
    start4(2) = 1
    start4(3) = 1
    start4(4) = nout

    count4(1) = nakx
    count4(2) = naky
    count4(3) = nspec
    count4(4) = 1

    starty(1) = 1
    starty(2) = nout

    county(1) = naky
    county(2) = 1

    startx(1) = 1
    startx(2) = nout

    countx(1) = nakx
    countx(2) = 1

    starts(1) = 1
    starts(2) = nout

    counts(1) = nspec
    counts(2) = 1

    starth(1) = 1 
    starth(2) = 1
    starth(3) = nout

    counth(1) = nspec
    counth(2) = 7
    counth(3) = 1

    if (use_Phi) then

       if (nakx > 1) then
          do it = 1, nakx
             field2_by_kx(it) = sum(phi2_by_mode(it,:)*fluxfac(:))
          end do
          status = netcdf_put_vara(ncid, phi2_by_kx_id, startx, countx, field2_by_kx)
       end if

       if (naky > 1) then
          do ik = 1, naky
             field2_by_ky(ik) = sum(phi2_by_mode(:,ik)*fluxfac(ik))
          end do
          status = netcdf_put_vara(ncid, phi2_by_ky_id, starty, county, field2_by_ky)
       end if

       call c2r (phi0, ri2) 
       status = netcdf_put_vara(ncid, phi0_id, start0, count0, ri2)
!       call c2r (phiavg, ri2) 
!       status = netcdf_put_vara(ncid, phiavg_id, start0, count0, ri2)
       status = netcdf_put_vara(ncid, phi2_by_mode_id, start, count, phi2_by_mode)
       status = netcdf_put_var1(ncid, phi2_id, nout, phi2)
    end if

    if (use_Apar) then

       if (nakx > 1) then
          do it = 1, nakx
             field2_by_kx(it) = sum(apar2_by_mode(it,:)*fluxfac(:))
          end do
          status = netcdf_put_vara(ncid, apar2_by_kx_id, startx, countx, field2_by_kx)
       end if

       if (naky > 1) then
          do ik = 1, naky
             field2_by_ky(ik) = sum(apar2_by_mode(:,ik)*fluxfac(ik))
          end do
          status = netcdf_put_vara(ncid, apar2_by_ky_id, starty, county, field2_by_ky)
       end if

       call c2r (apar0, ri2) 
       status = netcdf_put_vara(ncid, apar0_id, start0, count0, ri2)
       status = netcdf_put_vara(ncid, apar2_by_mode_id, start, count, apar2_by_mode)
       status = netcdf_put_var1(ncid, apar2_id, nout, apar2)
    end if

    if (use_Bpar) then

       if (nakx > 1) then
          do it = 1, nakx
             field2_by_kx(it) = sum(bpar2_by_mode(it,:)*fluxfac(:))
          end do
          status = netcdf_put_vara(ncid, bpar2_by_kx_id, startx, countx, field2_by_kx)
       end if

       if (naky > 1) then
          do ik = 1, naky
             field2_by_ky(ik) = sum(bpar2_by_mode(:,ik)*fluxfac(ik))
          end do
          status = netcdf_put_vara(ncid, bpar2_by_ky_id, starty, county, field2_by_ky)
       end if

       call c2r (bpar0, ri2) 
       status = netcdf_put_vara(ncid, bpar0_id, start0, count0, ri2)
       status = netcdf_put_vara(ncid, bpar2_by_mode_id, start, count, bpar2_by_mode)
       status = netcdf_put_var1(ncid, bpar2_id, nout, bpar2)
    end if
        
    if (write_hrate) then
       status = netcdf_put_var1(ncid, h_energy_id,     nout, h%energy)
       status = netcdf_put_var1(ncid, h_energy_dot_id, nout, h%energy_dot)
       status = netcdf_put_var1(ncid, h_antenna_id,    nout, h%antenna)
       status = netcdf_put_var1(ncid, h_eapar_id,      nout, h%eapar)
       status = netcdf_put_var1(ncid, h_ebpar_id,      nout, h%ebpar)

       status = netcdf_put_vara (ncid, h_delfs2_id,    starts, counts, h%delfs2)
       status = netcdf_put_vara (ncid, h_hs2_id,       starts, counts, h%hs2)
       status = netcdf_put_vara (ncid, h_phis2_id,     starts, counts, h%phis2)
       status = netcdf_put_vara (ncid, h_hypervisc_id, starts, counts, h%hypervisc)
       status = netcdf_put_vara (ncid, h_hyperres_id,  starts, counts, h%hyperres)
       status = netcdf_put_vara (ncid, h_collisions_id,starts, counts, h%collisions)
       status = netcdf_put_vara (ncid, h_gradients_id, starts, counts, h%gradients)
       status = netcdf_put_vara (ncid, h_imp_colls_id, starts, counts, h%imp_colls)
       status = netcdf_put_vara (ncid, h_hypercoll_id, starts, counts, h%hypercoll)
       status = netcdf_put_vara (ncid, h_heating_id,   starts, counts, h%heating)

       status = netcdf_put_vara (ncid, hk_energy_id,     start, count, hk%energy)
       status = netcdf_put_vara (ncid, hk_energy_dot_id, start, count, hk%energy_dot)
       status = netcdf_put_vara (ncid, hk_antenna_id,    start, count, hk%antenna)
       status = netcdf_put_vara (ncid, hk_eapar_id,      start, count, hk%eapar)
       status = netcdf_put_vara (ncid, hk_ebpar_id,      start, count, hk%ebpar)

       call hk_repack (hk, 1, tmps)
       status = netcdf_put_vara (ncid, hk_hypervisc_id, start4, count4, tmps)

       call hk_repack (hk, 2, tmps)
       status = netcdf_put_vara (ncid, hk_hyperres_id,  start4, count4, tmps)

       call hk_repack (hk, 3, tmps)
       status = netcdf_put_vara (ncid, hk_collisions_id,start4, count4, tmps)

       call hk_repack (hk, 4, tmps)
       status = netcdf_put_vara (ncid, hk_gradients_id, start4, count4, tmps)

       call hk_repack (hk, 5, tmps)
       status = netcdf_put_vara (ncid, hk_heating_id,   start4, count4, tmps)

       call hk_repack (hk, 6, tmps)
       status = netcdf_put_vara (ncid, hk_hypercoll_id, start4, count4, tmps)

       call hk_repack (hk, 7, tmps)
       status = netcdf_put_vara (ncid, hk_delfs2_id,    start4, count4, tmps)

       call hk_repack (hk, 8, tmps)
       status = netcdf_put_vara (ncid, hk_hs2_id,       start4, count4, tmps)

       call hk_repack (hk, 9, tmps)
       status = netcdf_put_vara (ncid, hk_phis2_id,     start4, count4, tmps)

       call hk_repack (hk, 10, tmps)
       status = netcdf_put_vara (ncid, hk_imp_colls_id, start4, count4, tmps)
    end if

    if (write_omega) then
       do it = 1, nakx
          tmp(it, :) = omega(it, :)
       end do
       
       call c2r (tmp, ri2) 
       status = netcdf_put_vara(ncid, omega_id, start0, count0, ri2)
       
       do it = 1, nakx
          tmp(it, :) = omegaavg(it, :)
       end do
       
       call c2r (tmp, ri2) 
       status = netcdf_put_vara(ncid, omegaavg_id, start0, count0, ri2)
    end if

!    status = netcdf_put_vara(ncid, phtot_id, start, count, phitot)
    
    if (mod(nout, 10) == 0) status = nf_sync (ncid)

  end subroutine nc_loop

  ! added by EAB on 03/05/04
  subroutine nc_loop_movie (nout_movie, time, yxphi, yxapar, yxbpar)
    use agk_layouts, only: yxf_lo
    use theta_grid, only: ntgrid
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use netcdf_mod, only: netcdf_put_var1, netcdf_put_vara
    integer, intent (in) :: nout_movie
    real, intent (in) :: time
    real, dimension (:,:,:), intent (in):: yxphi, yxapar, yxbpar 
    integer :: status
    integer, dimension (4) :: start, count

      
    status = netcdf_put_var1(ncid_movie, time_movie_id, nout_movie, time)
    start(1) = 1
    start(2) = 1
    start(3) = 1
    start(4) = nout_movie
    count(1) = yxf_lo%nx
    count(2) = yxf_lo%ny
    count(3) = 2*ntgrid+1
    count(4) = 1


    if (use_Phi) then
       status = netcdf_put_vara(ncid_movie, phi_by_xmode_id, start, count, yxphi)
    end if
!    if (use_Apar) then
       status = netcdf_put_vara(ncid_movie, apar_by_xmode_id, start, count, yxapar)
!    end if

    if (use_Bpar) then
       status = netcdf_put_vara(ncid_movie, bpar_by_xmode_id, start, count, yxbpar)
    end if

    if (mod(nout_movie, 10) == 0) status = nf_sync (ncid_movie)


  end subroutine nc_loop_movie

  subroutine nc_species
    
    use run_parameters, only: beta
    use netcdf_mod, only: netcdf_put_var
    use species, only: spec
    real :: betatot
    integer :: status

    status = netcdf_put_var (ncid, charge_id, spec%z)
    status = netcdf_put_var (ncid, mass_id, spec%mass)
    status = netcdf_put_var (ncid, dens_id, spec%dens)
    status = netcdf_put_var (ncid, temp_id, spec%temp)
    status = netcdf_put_var (ncid, tprim_id, spec%tprim)
    status = netcdf_put_var (ncid, fprim_id, spec%fprim)
    status = netcdf_put_var (ncid, uprim_id, spec%uprim)
    status = netcdf_put_var (ncid, nu_id, spec%nu)
    status = netcdf_put_var (ncid, spec_type_id, spec%type)

  end subroutine nc_species

  subroutine nc_geo

    use netcdf_mod, only: netcdf_put_var
    use theta_grid, only: gradpar, jacob
    integer :: status

    status = netcdf_put_var (ncid, gradpar_id, gradpar)
    status = netcdf_put_var (ncid, jacob_id, jacob)

  end subroutine nc_geo

end module agk_io

