module agk_diagnostics

  use agk_heating, only: heating_diagnostics,dens_vel_diagnostics

  implicit none

  public :: init_agk_diagnostics
  public :: finish_agk_diagnostics
  public :: loop_diagnostics

  interface get_vol_average
     module procedure get_vol_average_one, get_vol_average_all
  end interface

  ! knobs
  real :: omegatol, omegatinst
  logical :: print_line, print_flux_line, print_summary
  logical :: write_linear, write_nonlin
  logical :: write_omega, write_omavg, write_ascii
  logical :: write_gs
  logical :: write_g, write_gg
  logical :: write_vspace_slices
  logical :: write_epartot
  logical :: write_verr    ! MAB
  logical :: write_final_fields
  logical :: write_final_moments
  logical :: write_final_epar, write_kpar
  logical :: write_hrate, write_lorentzian
  logical :: write_adapt_hc       !Write values for adaptive hypercollisionality
  logical :: write_nl_flux, write_Epolar
  logical :: exit_when_converged
  logical :: make_movie
  logical :: save_for_restart
!>GGH
  logical :: write_density_velocity
  logical :: write_jext=.false.
!<GGH

  integer :: nwrite, igomega, nmovie
  integer :: navg, nsave

  ! internal
  logical :: write_any, write_any_fluxes, dump_any
  logical, private :: initialized = .false.

  integer :: out_unit, kp_unit, heat_unit, polar_raw_unit, polar_avg_unit
  integer, dimension (:), allocatable :: heat_unit2
  integer :: dv_unit, jext_unit   !GGH Additions
  integer :: res_unit  ! MAB
  integer, dimension(:), allocatable :: ahc_unit    !Adaptive HC unit for up to four species
  logical :: nuh_changed   !Flag =T if any nu_h is changed by adaptive hypercollisions

  complex, dimension (:,:,:), allocatable :: omegahist
  ! (navg,nakx,naky)
  type (heating_diagnostics), dimension(:,:,:), allocatable, save :: hk_hist
  type (heating_diagnostics), dimension(:,:), allocatable, save :: hk
  type (heating_diagnostics), dimension(:), allocatable, save :: h_hist
  type (heating_diagnostics), save :: h

  !GGH Density/velocity pertubration diagnostics
  type (dens_vel_diagnostics), dimension(:), allocatable :: dv_hist
  type (dens_vel_diagnostics), dimension(:,:,:), allocatable :: dvk_hist
  !GGH J_external
  real, dimension(:,:,:), allocatable ::  j_ext_hist
  !Polar spectrum variables
  integer :: nbx                                   !Number of polar bins
  real, dimension(:), allocatable :: kpbin,ebincorr
  real, dimension(:), allocatable :: numavg,kpavg
  real, dimension(:,:), pointer :: ebinarray => null ()
  real, dimension(:,:), pointer :: eavgarray => null ()
  real, dimension(:,:), allocatable :: etmp
  integer, dimension(:,:), allocatable :: polar_index
  integer, dimension(:), allocatable :: polar_avg_index
  integer, parameter :: iefluc = 1        !Total fluctuating energy
  integer, parameter :: ieapar = 2        !Energy in A_parallel
  integer, parameter :: iebpar = 3        !Energy in B_parallel
  integer, parameter :: iephis2= 4        !Energy in q_s^ n_s Phi^2/T_s
  integer, parameter :: iehs2   = 5       !Energy in hs^2
  integer, parameter :: iedelfs2= 6       !Energy in del f_s^2
  integer, dimension(:,:,:), allocatable :: ahc_index
  !GGH Adaptive Hypercollisionality variables
  real, dimension(:), allocatable :: pk1,pk2,ek,etot,wnl,kphc,bk
  real, dimension(:), allocatable :: fm1,fm2,ft !Measures for adjusting nu_h


  real, dimension (:,:,:,:), allocatable ::  qheat, qmheat, qbheat
  ! (nakx,naky,nspec,3)

  real, dimension (:,:,:), allocatable ::  pflux, pmflux, pbflux
  ! (nakx,naky,nspec)

  real, dimension (:,:,:), allocatable ::  vflux, vmflux, vbflux
  ! (nakx,naky,nspec)

  integer :: ntg

contains

  subroutine init_agk_diagnostics (list, nstep)
    use theta_grid, only: init_theta_grid
    use kgrids, only: init_kgrids, nakx, naky
    use run_parameters, only: init_run_parameters
    use species, only: init_species, nspec, adapt_hc_any
    use dist_fn, only: init_dist_fn
    use agk_io, only: init_agk_io
    use agk_heating, only: init_htype,init_dvtype
    use mp, only: broadcast, proc0
    use le_grids, only: init_weights
    implicit none
    logical, intent (in) :: list
    integer, intent (in) :: nstep
    integer :: ik, it, nmovie_tot

    if (initialized) return
    initialized = .true.

    call init_theta_grid
    call init_kgrids
    call init_run_parameters
    call init_species
    call init_dist_fn

    call real_init (list)
    call broadcast (navg)
    call broadcast (nwrite)
    call broadcast (nmovie)
    call broadcast (nsave)
    call broadcast (write_any)
    call broadcast (write_any_fluxes)
    call broadcast (write_nl_flux)
    call broadcast (write_omega)
    call broadcast (dump_any)
    call broadcast (make_movie)
    call broadcast (save_for_restart)
    call broadcast (write_gs)
    call broadcast (write_g)
    call broadcast (write_gg)
    call broadcast (write_vspace_slices)

    call broadcast (ntg)
    call broadcast (write_hrate)
    call broadcast (write_density_velocity)
    call broadcast (write_adapt_hc)
    call broadcast (write_Epolar)
    call broadcast (write_lorentzian)
    call broadcast (write_verr)   ! MAB

    nmovie_tot = nstep/nmovie

!>MAB
! initialize weights for less accurate integrals used
! to provide an error estimate for v-space integrals (energy and untrapped)
    if (write_verr .and. proc0) call init_weights
!<MAB

! allocate heating diagnostic data structures
    if (write_hrate) then
       allocate (h_hist(0:navg-1))
       call init_htype (h_hist,  nspec)

       allocate (hk_hist(nakx,naky,0:navg-1))
       call init_htype (hk_hist, nspec)

       call init_htype (h,  nspec)
       allocate (hk(nakx, naky))
       call init_htype (hk, nspec)

    else
       allocate (h_hist(0))
       allocate (hk(1,1))
       allocate (hk_hist(1,1,0))
    end if
       
!GGH Allocate density and velocity perturbation diagnostic structures
    if (write_density_velocity) then
       allocate (dv_hist(0:navg-1))
       call init_dvtype (dv_hist,  nspec)

       allocate (dvk_hist(nakx,naky,0:navg-1))
       call init_dvtype (dvk_hist, nspec)
    end if
       
!GGH Allocate density and velocity perturbation diagnostic structures
    if (write_jext) allocate (j_ext_hist(nakx, naky,0:navg-1)) 

!Initialize polar spectrum diagnostic
    if (write_Epolar .and. proc0) call init_polar_spectrum

!Initialize adaptive hypercollisionality diagnostic
    if ((adapt_hc_any .or. write_adapt_hc) .and. proc0) call init_adapt_hc

    call init_agk_io (write_nl_flux, write_omega, write_hrate, make_movie, nmovie_tot)
    
  end subroutine init_agk_diagnostics
 
  subroutine real_init (list)
    use run_parameters, only: use_Apar
    use file_utils, only: open_output_file, get_unused_unit
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, aky, akx
!    use agk_layouts, only: yxf_lo
    use species, only: nspec
    use mp, only: proc0
    use constants
    implicit none
    logical, intent (in) :: list
    character(20) :: datestamp, timestamp, zone, suffix
    integer :: ig, ik, it, is

    call read_parameters (list)
    if (proc0) then
       if (write_ascii) then
          call open_output_file (out_unit, ".out")
!          if (write_kpar) call open_output_file (kp_unit, ".kp")
       end if
       
       if (write_hrate .and. write_ascii) then
          call open_output_file (heat_unit, ".heat")
          allocate(heat_unit2(1:nspec))
          do is=1,nspec
             write(suffix,'(a5,i2.2)')".heat",is
             call open_output_file (heat_unit2(is),suffix)
          enddo
       end if

       !Adaptive hypercollisionality output
       if (write_adapt_hc .and. write_ascii) then
          !Allocate variable for unit numbers
          allocate(ahc_unit(1:nspec))
          do is= 1, nspec
             write(suffix,'(a4,i2.2)')".ahc",is
             call open_output_file (ahc_unit(is), suffix)
          enddo
       endif

       !GGH Density and velocity perturbations
       if (write_density_velocity .and. write_ascii) then
          call open_output_file (dv_unit, ".dv")
       end if

       !GGH J_external, only if A_parallel is being calculated.
       if (write_jext .and. use_Apar) then
          if (write_ascii) then
             call open_output_file (jext_unit, ".jext")
          end if
       else
          write_jext = .false.
       end if

       if (write_Epolar .and. write_ascii) then
          call open_output_file (polar_raw_unit, ".kspec_raw")
          call open_output_file (polar_avg_unit, ".kspec_avg")
       end if

!>MAB
       if (write_verr .and. write_ascii) then
          call open_output_file (res_unit, ".vres")
       end if
!<MAB

       if (write_ascii) then
          write (unit=out_unit, fmt="('AstroGK')")
          datestamp(:) = ' '
          timestamp(:) = ' '
          zone(:) = ' '
          call date_and_time (datestamp, timestamp, zone)
          write (unit=out_unit, fmt="('Date: ',a,' Time: ',a,1x,a)") &
               trim(datestamp), trim(timestamp), trim(zone)
       end if
       
       allocate (omegahist(0:navg-1,nakx,naky))
       omegahist = 0.0
    end if

    allocate (pflux (nakx,naky,nspec)) ; pflux = 0.
    allocate (qheat (nakx,naky,nspec,3)) ; qheat = 0.
    allocate (vflux (nakx,naky,nspec)) ; vflux = 0.
    allocate (pmflux(nakx,naky,nspec)) ; pmflux = 0.
    allocate (qmheat(nakx,naky,nspec,3)) ; qmheat = 0.
    allocate (vmflux(nakx,naky,nspec)) ; vmflux = 0.
    allocate (pbflux(nakx,naky,nspec)) ; pbflux = 0.
    allocate (qbheat(nakx,naky,nspec,3)) ; qbheat = 0.
    allocate (vbflux(nakx,naky,nspec)) ; vbflux = 0.
       
  end subroutine real_init

  subroutine read_parameters (list)
    use file_utils, only: input_unit, run_name, input_unit_exist
    use theta_grid, only: nperiod, ntheta
    use kgrids, only: box, nx, ny
    use mp, only: proc0
    implicit none
    integer :: in_file
    logical, intent (in) :: list
    logical :: exist
    namelist /diagnostics/ &
         print_line, print_flux_line, &
         write_linear, write_nonlin, &
         write_omega, write_omavg, write_ascii, write_kpar, &
         write_gs, write_g, write_gg, write_vspace_slices, &
         write_hrate, write_density_velocity, write_epartot, &
         write_final_fields, write_final_epar, write_final_moments, &
         write_Epolar, write_nl_flux, &
         nwrite, nmovie, nsave, navg, omegatol, omegatinst, igomega, write_lorentzian, &
         exit_when_converged, make_movie, save_for_restart, &
         write_adapt_hc, &
         write_verr   ! MAB

    if (proc0) then
       print_line = .true.
       print_flux_line = .false.
       write_linear = .true.
       write_nonlin = .true.
       write_kpar = .false.
       write_hrate = .false.
       write_adapt_hc = .false.
       write_density_velocity = .false.
       write_Epolar = .false.
       write_gs = .false.
       write_g = .false.
       write_gg = .false.
       write_vspace_slices = .false.
       write_lorentzian = .false.
       write_omega = .false.
       write_ascii = .true.
       write_omavg = .false.
       write_epartot = .false.
       write_nl_flux = .false.
       write_final_moments = .false.
       write_final_fields = .false.
       write_final_epar = .false.
       write_verr = .false.   ! MAB
       nwrite = 100
       nmovie = 1000
       navg = 100
       nsave = -1
       omegatol = 1e-3
       omegatinst = 1.0
       igomega = 0
       exit_when_converged = .true.
       make_movie = .false.
       save_for_restart = .false.
       in_file = input_unit_exist ("diagnostics", exist)
       if (exist) read (unit=input_unit("diagnostics"), nml=diagnostics)

       print_summary = (list .and. (print_line .or. print_flux_line)) 

       if (list) then
          print_line = .false.
          print_flux_line = .false.
       end if

       if (.not. save_for_restart) nsave = -1

! Only calculate polar integrals in box layout
       write_Epolar = write_Epolar .and. box

! Disable polar integrals if nx /= ny
       if (nx /= ny) write_Epolar = .false.

       write_any = write_linear .or. write_omega     .or. write_omavg &
            .or. write_nonlin .or. write_nl_flux .or. write_Epolar &
            .or. write_kpar   .or. write_hrate     .or. write_lorentzian  .or. write_gs &
            .or. write_density_velocity .or. write_adapt_hc 
       write_any_fluxes =  write_nonlin .or. print_flux_line .or. write_nl_flux
       dump_any = make_movie .or. print_summary

       ntg = ntheta/2 + (nperiod-1)*ntheta
    end if 
 end subroutine read_parameters

  subroutine finish_agk_diagnostics (istep)
    use file_utils, only: open_output_file, close_output_file, get_unused_unit
    use mp, only: proc0, broadcast, nproc, iproc, sum_reduce, barrier
    use species, only: nspec, spec, adapt_hc_any
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use theta_grid, only: ntgrid, theta, delthet, jacob, gradpar, nperiod
    use kgrids, only: naky, nakx, nx, ny, aky, akx
    use le_grids, only: nlambda, negrid, al, delal
    use le_grids, only: e, dele
    use fields_arrays, only: phi, apar, bpar, phinew, aparnew
    use dist_fn, only: get_epar, getmoms, par_spectrum
    use dist_fn, only: write_f, write_vp
    use dist_fn_arrays, only: g, gnew
    use agk_layouts, only: xxf_lo
    use agk_transforms, only: transform2, inverse2
    use agk_save, only: agk_save_for_restart
    use constants
    use agk_time, only: time, dtime
    use agk_io, only: nc_final_fields, nc_final_epar
    use agk_io, only: nc_final_moments, nc_finish
!    use antenna, only: dump_ant_amp
    use splines, only: fitp_surf1, fitp_surf2
!    use agk_dist_io, only: write_dist
! TT>
    use agk_heating, only: del_htype
! <TT
    use nonlinear_terms, only: nonlin    !GGH
    implicit none
    integer, intent (in) :: istep
    integer :: ig, ik, it, il, ie, is, unit, ierr
    real, dimension (:), allocatable :: total
    real, dimension (:,:,:), allocatable :: xphi, xapar, xbpar
    real, dimension (:,:,:), allocatable :: bxf, byf, vxf, vyf, bxfsavg, byfsavg
    real, dimension (:,:,:), allocatable :: bxfs, byfs, vxfs, vyfs, rvx, rvy, rx, ry
    complex, dimension (:,:,:), allocatable :: bx, by, vx, vy, vx2, vy2
    complex, dimension (:,:,:), allocatable :: phi2, apar2, bpar2, epar
    complex, dimension (:,:,:,:), allocatable :: ntot, density, upar, tpar, tperp
    real, dimension (:), allocatable :: dl_over_b
    complex, dimension (nakx, naky) :: phi0
    real, dimension (nakx, naky) :: phi02
    real, dimension (nspec) :: weights
    real, dimension (2*ntgrid) :: kpar
    real, dimension (:), allocatable :: xx4, yy4, dz
    real, dimension (:,:), allocatable :: bxs, bys, vxs, vys
    real, dimension (:,:), allocatable :: bxsavg, bysavg
    real, dimension (:), allocatable :: stemp, zx1, zxm, zy1, zyn, xx, yy
    real :: zxy11, zxym1, zxy1n, zxymn, L_x, L_y, rxt, ryt, bxt, byt
    integer :: istatus, nnx, nny, nnx4, nny4, ulim, llim, iblock, i, g_unit
    logical :: last = .true.
    !GGH
    real :: nt1,dens1,upar1,tpar1,tperp1
    real :: fac2                                     !Factor
    real, dimension (:), allocatable :: wgt

    !Set up weighting factors for z-sums on proc0
    if (proc0) then
       allocate (wgt(-ntgrid:ntgrid))
       wgt = 0.
       do ig=-ntgrid,ntgrid-1
          wgt(ig) = delthet*jacob
       end do
       wgt = wgt/sum(wgt)         
    endif

!    TT changed MAB's version to use phi and bpar from arrays
!    if (write_g) call write_f (last, phi, bpar)
    if (write_g) call write_f (last)

    !Write out v-space slices for each Fourier mode
    if (write_vspace_slices) call write_vp

!    if (write_gg) call write_dist (g)

    phi0 = 1.

    if (proc0) then
       if (write_ascii) call close_output_file (out_unit)
       if (write_ascii .and. write_hrate) then
          call close_output_file (heat_unit)
          do is= 1, nspec
             call close_output_file (heat_unit2(is))
          enddo
       endif
       if (write_adapt_hc .and. write_ascii) then
          do is= 1, nspec
             call close_output_file (ahc_unit(is))
          enddo
          !Deallocate variable for ahc_unit
          deallocate(ahc_unit)
       endif
       if (write_ascii .and. write_density_velocity) call close_output_file (dv_unit)
       if (write_ascii .and. write_jext) call close_output_file (jext_unit)
       if (write_ascii .and. write_Epolar) then
          call close_output_file (polar_raw_unit)
          call close_output_file (polar_avg_unit)
       endif
!>MAB
       if (write_ascii .and. write_verr) then
          call close_output_file (res_unit)
       end if
!<MAB

       !Finish polar spectrum diagnostic (deallocate variables)
       if (write_Epolar) call finish_polar_spectrum

       !Finish adaptive hypercollisionality (deallocate variables)
       if ((adapt_hc_any .or. write_adapt_hc) .and. proc0) call finish_adapt_hc

       if (write_final_fields) then
          if (write_ascii) then
             call open_output_file (unit, ".fields")
             do ik = 1, naky
                do it = 1, nakx
                   do ig = -ntgrid, ntgrid
                      write (unit, "(9(1x,e12.5))") &
                           theta(ig), aky(ik), akx(it), &
                           phi(ig,it,ik), apar(ig,it,ik), bpar(ig,it,ik)
                   end do
                   write (unit, "()")
                end do
             end do
             call close_output_file (unit)
          end if
          call nc_final_fields
       end if
       
       if (write_kpar) then

          allocate ( phi2(-ntgrid:ntgrid,nakx,naky)) ;  phi2 = 0.
          allocate (apar2(-ntgrid:ntgrid,nakx,naky)) ; apar2 = 0.
          allocate (bpar2(-ntgrid:ntgrid,nakx,naky)) ; bpar2 = 0.

          if (use_Phi) call par_spectrum(phi, phi2)
          if (use_Apar) call par_spectrum(apar, apar2)
          if (use_Bpar) call par_spectrum(bpar, bpar2)

          call open_output_file (unit, ".kpar")
          do ig = 1, ntgrid
             kpar(ig) = (ig-1)*gradpar/real(2*nperiod-1)
             kpar(2*ntgrid-ig+1)=-(ig)*gradpar/real(2*nperiod-1)
          end do
          do ik = 1, naky
             do it = 1, nakx
                do ig = ntgrid+1,2*ntgrid
                   write (unit, "(9(1x,e12.5))") &
                        kpar(ig), aky(ik), akx(it), &
                        phi2(ig-ntgrid-1,it,ik), &
                        apar2(ig-ntgrid-1,it,ik), &
                        bpar2(ig-ntgrid-1,it,ik)                        
                end do
                do ig = 1, ntgrid
                   write (unit, "(9(1x,e12.5))") &
                        kpar(ig), aky(ik), akx(it), &
                        phi2(ig-ntgrid-1,it,ik), &
                        apar2(ig-ntgrid-1,it,ik), &
                        bpar2(ig-ntgrid-1,it,ik)
                end do
                write (unit, "()")
             end do
          end do
          call close_output_file (unit)
          deallocate (phi2, apar2, bpar2)
       end if

       if (write_final_epar) then
          allocate (epar(-ntgrid:ntgrid,nakx,naky))   ; epar = 0.

          call get_epar (phi, apar, phinew, aparnew, epar)
          if (write_ascii) then
             call open_output_file (unit, ".epar")
             do ik = 1, naky
                do it = 1, nakx
                   do ig = -ntg, ntg-1
                      write (unit, "(5(1x,e12.5))") &
                           theta(ig), aky(ik), akx(it), epar(ig,it,ik)
                   end do
                   write (unit, "()")
                end do
             end do
             call close_output_file (unit)
          end if
          call nc_final_epar (epar  )
          deallocate (epar)
       end if
    end if

! TT>
    if (write_hrate) then
       call del_htype (h)
       call del_htype (h_hist)
       call del_htype (hk)
       call del_htype (hk_hist)
       deallocate (h_hist, hk, hk_hist)
    end if
! <TT

    call broadcast (write_final_moments)
    if (write_final_moments) then

       allocate (ntot(-ntgrid:ntgrid,nakx,naky,nspec))
       allocate (density(-ntgrid:ntgrid,nakx,naky,nspec))
       allocate (upar(-ntgrid:ntgrid,nakx,naky,nspec))
       allocate (tpar(-ntgrid:ntgrid,nakx,naky,nspec))
       allocate (tperp(-ntgrid:ntgrid,nakx,naky,nspec))
       call getmoms (phinew, ntot, density, upar, tpar, tperp)

       if (proc0) then
          if (write_ascii) then
             call open_output_file (unit, ".moments")
             phi0 = 1.
             do is  = 1, nspec
                nt1=0.;dens1=0.;upar1=0.;tpar1=0.;tperp=0.
                do ik = 1, naky
                   fac2 = 0.5
                   if (aky(ik) < epsilon(0.0)) fac2 = 1.0
                   do it = 1, nakx
                     !Skip mean value (k=0) components
                     if (nonlin .and. it == 1 .and. ik == 1) cycle
                      do ig = -ntg, ntg
                         write (unit, "(14(1x,e12.5))") &
                              theta(ig), aky(ik), akx(it), &
                              ntot(ig,it,ik,is)/phi0(it,ik), &
                              density(ig,it,ik,is)/phi0(it,ik), &
                              upar(ig,it,ik,is)/phi0(it,ik), &
                              tpar(ig,it,ik,is)/phi0(it,ik), &
                              tperp(ig,it,ik,is)/phi0(it,ik), &
                              real(is)
                         !GGH-Sum all of the values over ntgrid
                         nt1=nt1+abs(ntot(ig,it,ik,is))*wgt(ig)*fac2
                         dens1=dens1+abs(density(ig,it,ik,is))*wgt(ig)*fac2
                         upar1=upar1+abs(upar(ig,it,ik,is))*wgt(ig)*fac2
                         tpar1=tpar1+abs(tpar(ig,it,ik,is))*wgt(ig)*fac2
                         tperp1=tperp1+abs(tperp(ig,it,ik,is))*wgt(ig)*fac2
                         
                      end do
                      write(unit,"(5es12.4)")nt1,dens1,upar1,tpar1,tperp1
                      write (unit, "()")
                   end do
                end do
             end do
             call close_output_file (unit)          
          end if
          call nc_final_moments (ntot, density, upar, tpar, tperp)

          if (write_ascii) then
             call open_output_file (unit, ".mom2")
             phi0 = 1.
             phi02=real(phi0*conjg(phi0))
             do is  = 1, nspec
                do ik = 1, naky
                   do it = 1, nakx
                      do ig = -ntg, ntg
                         write (unit, "(14(1x,e12.5))") &
                              theta(ig), aky(ik), akx(it), &
                              real(ntot(ig,it,ik,is)*conjg(ntot(ig,it,ik,is)))/phi02(it,ik), &
                              real(density(ig,it,ik,is)*conjg(density(ig,it,ik,is)))/phi02(it,ik), &
                              real(upar(ig,it,ik,is)*conjg(upar(ig,it,ik,is)))/phi02(it,ik), &
                              real(tpar(ig,it,ik,is)*conjg(tpar(ig,it,ik,is)))/phi02(it,ik), &
                              real(tperp(ig,it,ik,is)*conjg(tperp(ig,it,ik,is)))/phi02(it,ik), &
                              real(is)
                      end do
                      write (unit, "()")
                   end do
                end do
             end do

             call close_output_file (unit)          
             call open_output_file (unit, ".amoments")
             write (unit,*) 'type    kx     re(phi)    im(phi)    re(ntot)   im(ntot)   ',&
                  &'re(dens)   im(dens)   re(upar)   im(upar)   re(tpar)',&
                  &'   im(tpar)   re(tperp)  im(tperp)'
             
             allocate (dl_over_b(-ntgrid:ntgrid))
             
             dl_over_b = delthet*jacob
             dl_over_b = dl_over_b / sum(dl_over_b)
             
             do is  = 1, nspec
                do it = 2, nakx/2+1
                   write (unit, "(i2,14(1x,e10.3))") spec(is)%type, akx(it), &
                        sum( phinew(:,it,1)   *dl_over_b), &
                        sum(   ntot(:,it,1,is)*dl_over_b), &
                        sum(density(:,it,1,is)*dl_over_b), &
                        sum(   upar(:,it,1,is)*dl_over_b), &
                        sum(   tpar(:,it,1,is)*dl_over_b), &
                        sum(  tperp(:,it,1,is)*dl_over_b)
                end do
             end do
             deallocate (dl_over_b)
             call close_output_file (unit)          
          end if
       end if
       deallocate (ntot, density, upar, tpar, tperp)
    end if

    if (save_for_restart) then
       call agk_save_for_restart (gnew, time, dtime, istatus, &
            use_Phi, use_Apar, use_Bpar, .true.)
    end if

    call nc_finish

    !GGH- Commented this out as unnecessary 07 Jun 07
!    if (proc0) call dump_ant_amp

    if (write_gs) then
       nny = 2*ny
       nnx = 2*nx
       nnx4 = nnx+4
       nny4 = nny+4

       allocate (bx(-ntgrid:ntgrid,nakx,naky))
       allocate (by(-ntgrid:ntgrid,nakx,naky))
       allocate (vx(-ntgrid:ntgrid,nakx,naky))
       allocate (vy(-ntgrid:ntgrid,nakx,naky))

       do ik=1,naky
          do it=1,nakx
             do ig=-ntgrid, ntgrid
                bx(ig,it,ik) =  zi*aky(ik)*apar(ig,it,ik)
                by(ig,it,ik) = -zi*akx(it)*apar(ig,it,ik)
                vx(ig,it,ik) = -zi*aky(ik)*phi(ig,it,ik)
                vy(ig,it,ik) =  zi*akx(it)*phi(ig,it,ik)
             end do
          end do
       end do

       allocate (bxf(nnx,nny,-ntgrid:ntgrid))
       allocate (byf(nnx,nny,-ntgrid:ntgrid))
       allocate (vxf(nnx,nny,-ntgrid:ntgrid))
       allocate (vyf(nnx,nny,-ntgrid:ntgrid))

       call transform2 (bx, bxf, nny, nnx)
       call transform2 (by, byf, nny, nnx)
       call transform2 (vx, vxf, nny, nnx)
       call transform2 (vy, vyf, nny, nnx)
       
       ! fields come out as (x, y, z)

       deallocate (bx, by)

       allocate (   bxfs(nnx4, nny4, -ntgrid:ntgrid))
       allocate (   byfs(nnx4, nny4, -ntgrid:ntgrid))
       allocate (bxfsavg(nnx4, nny4, -ntgrid:ntgrid))
       allocate (byfsavg(nnx4, nny4, -ntgrid:ntgrid))
       allocate (   vxfs(nnx4, nny4, -ntgrid:ntgrid))
       allocate (   vyfs(nnx4, nny4, -ntgrid:ntgrid))

       do ig=-ntgrid,ntgrid
          do ik=1,2
             do it=3,nnx4-2
                bxfs(it,ik,ig) = bxf(it-2,nny-2+ik,ig)
                byfs(it,ik,ig) = byf(it-2,nny-2+ik,ig)
                vxfs(it,ik,ig) = vxf(it-2,nny-2+ik,ig)
                vyfs(it,ik,ig) = vyf(it-2,nny-2+ik,ig)

                bxfs(it,nny4-2+ik,ig) = bxf(it-2,ik,ig)
                byfs(it,nny4-2+ik,ig) = byf(it-2,ik,ig)
                vxfs(it,nny4-2+ik,ig) = vxf(it-2,ik,ig)
                vyfs(it,nny4-2+ik,ig) = vyf(it-2,ik,ig)
             end do
          end do
          do ik=3,nny4-2
             do it=3,nnx4-2
                bxfs(it,ik,ig) = bxf(it-2,ik-2,ig)
                byfs(it,ik,ig) = byf(it-2,ik-2,ig)
                vxfs(it,ik,ig) = vxf(it-2,ik-2,ig)
                vyfs(it,ik,ig) = vyf(it-2,ik-2,ig)
             end do
          end do
          do ik=1,nny4
             do it=1,2
                bxfs(it,ik,ig) = bxfs(nnx4-4+it,ik,ig)
                byfs(it,ik,ig) = byfs(nnx4-4+it,ik,ig)
                vxfs(it,ik,ig) = vxfs(nnx4-4+it,ik,ig)
                vyfs(it,ik,ig) = vyfs(nnx4-4+it,ik,ig)

                bxfs(nnx4-2+it,ik,ig) = bxfs(it+2,ik,ig)
                byfs(nnx4-2+it,ik,ig) = byfs(it+2,ik,ig)
                vxfs(nnx4-2+it,ik,ig) = vxfs(it+2,ik,ig)
                vyfs(nnx4-2+it,ik,ig) = vyfs(it+2,ik,ig)
             end do
          end do
       end do

       deallocate (vxf, vyf)

       allocate (xx4(nnx4), xx(nnx))
       allocate (yy4(nny4), yy(nny))
       
       L_x = 2.0*pi/akx(2)
       L_y = 2.0*pi/aky(2)

       do it = 1, nnx
          xx4(it+2) = real(it-1)*L_x/real(nnx)
          xx(it) = real(it-1)*L_x/real(nnx)
       end do
       do it=1,2
          xx4(it) = xx4(nnx4-4+it)-L_x
          xx4(nnx4-2+it) = xx4(it+2)+L_x
       end do

       do ik = 1, nny
          yy4(ik+2) = real(ik-1)*L_y/real(nny)
          yy(ik)    = real(ik-1)*L_y/real(nny)
       end do
       do ik=1,2
          yy4(ik) = yy4(nny4-4+ik)-L_y
          yy4(nny4-2+ik) = yy4(ik+2)+L_y
       end do

       allocate (dz(-ntgrid:ntgrid))
       dz = delthet*jacob

       allocate (   bxs(3*nnx4*nny4,-ntgrid:ntgrid)) ; bxs = 0.
       allocate (   bys(3*nnx4*nny4,-ntgrid:ntgrid)) ; bys = 0.
       allocate (   vxs(3*nnx4*nny4,-ntgrid:ntgrid)) ; vxs = 0.
       allocate (   vys(3*nnx4*nny4,-ntgrid:ntgrid)) ; vys = 0.

       allocate (bxsavg(3*nnx4*nny4,-ntgrid:ntgrid))
       allocate (bysavg(3*nnx4*nny4,-ntgrid:ntgrid))

       allocate (stemp(nnx4+2*nny4))
       allocate (zx1(nny4), zxm(nny4), zy1(nnx4), zyn(nnx4))

       do ig=-ntgrid, ntgrid
          call fitp_surf1(nnx4, nny4, xx, yy, bxfs(:,:,ig), &
               nnx4, zx1, zxm, zy1, zyn, zxy11, zxym1, zxy1n, zxymn, &
               255, bxs(:,ig), stemp, 1., ierr)

          call fitp_surf1(nnx4, nny4, xx, yy, byfs(:,:,ig), &
               nnx4, zx1, zxm, zy1, zyn, zxy11, zxym1, zxy1n, zxymn, &
               255, bys(:,ig), stemp, 1., ierr)

          call fitp_surf1(nnx4, nny4, xx, yy, vxfs(:,:,ig), &
               nnx4, zx1, zxm, zy1, zyn, zxy11, zxym1, zxy1n, zxymn, &
               255, vxs(:,ig), stemp, 1., ierr)

          call fitp_surf1(nnx4, nny4, xx, yy, vyfs(:,:,ig), &
               nnx4, zx1, zxm, zy1, zyn, zxy11, zxym1, zxy1n, zxymn, &
               255, vys(:,ig), stemp, 1., ierr)
       end do

       deallocate (zx1, zxm, zy1, zyn, stemp)

       do ig=-ntgrid, ntgrid-1
          bxsavg(:,ig) = 0.5*(bxs(:,ig)+bxs(:,ig+1))
          bysavg(:,ig) = 0.5*(bys(:,ig)+bys(:,ig+1))

          bxfsavg(:,:,ig) = 0.5*(bxfs(:,:,ig)+bxfs(:,:,ig+1))
          byfsavg(:,:,ig) = 0.5*(byfs(:,:,ig)+byfs(:,:,ig+1))
       end do

       ! now, integrate to find a field line

       allocate ( rx(nnx,nny,-ntgrid:ntgrid))
       allocate ( ry(nnx,nny,-ntgrid:ntgrid))
       allocate (rvx(nnx,nny,-ntgrid:ntgrid)) ; rvx = 0.
       allocate (rvy(nnx,nny,-ntgrid:ntgrid)) ; rvy = 0.

       do ik=1,nny
          do it=1,nnx
             rx(it,ik,-ntgrid) = xx(it)
             ry(it,ik,-ntgrid) = yy(ik)
          end do
       end do

       iblock = (nnx*nny-1)/nproc + 1
       llim = 1 + iblock * iproc
       ulim = min(nnx*nny, llim+iblock-1)

       do i=llim, ulim
          it = 1 + mod(i-1, nnx)
          ik = 1 + mod((i-1)/nnx, nny)
          
          ig = -ntgrid
          
          rxt = rx(it,ik,ig)
          ryt = ry(it,ik,ig)
          
          rvx(it,ik,ig) = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, vxfs(:,:,ig), nnx4, vxs(:,ig), 1.)
          rvy(it,ik,ig) = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, vyfs(:,:,ig), nnx4, vys(:,ig), 1.)
          
          do ig=-ntgrid,ntgrid-1
             
             bxt = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, bxfs(:,:,ig), nnx4, bxs(:,ig), 1.)
             byt = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, byfs(:,:,ig), nnx4, bys(:,ig), 1.)
             
             rxt = rx(it,ik,ig) + 0.5*dz(ig)*bxt  
             ryt = ry(it,ik,ig) + 0.5*dz(ig)*byt  
             
             if (rxt > L_x) rxt = rxt - L_x
             if (ryt > L_y) ryt = ryt - L_y
             
             if (rxt < 0.) rxt = rxt + L_x
             if (ryt < 0.) ryt = ryt + L_y
             
             bxt = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, bxfsavg(:,:,ig), nnx4, bxsavg(:,ig), 1.)
             byt = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, byfsavg(:,:,ig), nnx4, bysavg(:,ig), 1.)
             
             rxt = rx(it,ik,ig) + dz(ig)*bxt  
             ryt = ry(it,ik,ig) + dz(ig)*byt  
             
             if (rxt > L_x) rxt = rxt - L_x
             if (ryt > L_y) ryt = ryt - L_y
             
             if (rxt < 0.) rxt = rxt + L_x
             if (ryt < 0.) ryt = ryt + L_y
             
             rx(it,ik,ig+1) = rxt
             ry(it,ik,ig+1) = ryt
             
             rvx(it,ik,ig+1) = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, vxfs(:,:,ig+1), nnx4, vxs(:,ig+1), 1.)
             rvy(it,ik,ig+1) = fitp_surf2(rxt, ryt, nnx4, nny4, xx4, yy4, vyfs(:,:,ig+1), nnx4, vys(:,ig+1), 1.)
          end do
       end do

       deallocate (bxfs, byfs, bxfsavg, byfsavg, vxfs, vyfs)
       deallocate (rx, ry, bxs, bys, vxs, vys, bxsavg, bysavg)

       allocate (total(2*nnx*nny*(2*ntgrid+1)))

       i=1
       do ig=-ntgrid,ntgrid
          do ik=1,nny
             do it=1,nnx
                total(i) = rvx(it,ik,ig)
                total(i+1) = rvy(it,ik,ig)
                i = i + 2
             end do
          end do
       end do
       
       call sum_reduce(total, 0)

       i=1
       do ig=-ntgrid,ntgrid
          do ik=1,nny
             do it=1,nnx
                rvx(it,ik,ig) = total(i)
                rvy(it,ik,ig) = total(i+1)
                i = i + 2
             end do
          end do
       end do
       
       if (proc0) then
          call inverse2 (rvx, vx, nny, nnx)
          call inverse2 (rvy, vy, nny, nnx)
       
          allocate (vx2(-ntgrid:ntgrid,nakx,naky))
          allocate (vy2(-ntgrid:ntgrid,nakx,naky))

          call par_spectrum (vx, vx2)
          call par_spectrum (vy, vy2)

          call open_output_file (unit, ".gs")
          do ig = 1, ntgrid
             kpar(ig) = (ig-1)*gradpar/real(2*nperiod-1)
             kpar(2*ntgrid-ig+1)=-(ig)*gradpar/real(2*nperiod-1)
          end do
          do ik = 1, naky
             do it = 1, nakx
                do ig = ntgrid+1,2*ntgrid
                   write (unit, "(9(1x,e12.5))") &
                        kpar(ig), aky(ik), akx(it), &
                        real(vx2(ig-ntgrid-1,it,ik)), &
                        real(vy2(ig-ntgrid-1,it,ik))
                end do
                do ig = 1, ntgrid
                   write (unit, "(9(1x,e12.5))") &
                        kpar(ig), aky(ik), akx(it), &
                        real(vx2(ig-ntgrid-1,it,ik)), &
                        real(vy2(ig-ntgrid-1,it,ik))
                end do
                write (unit, "()")
             end do
          end do
          call close_output_file (unit)
          deallocate (vx2, vy2)
       end if

       deallocate (vx, vy, rvx, rvy)
       !GGH
       if (proc0) deallocate(wgt)

    end if
    
  end subroutine finish_agk_diagnostics

  subroutine loop_diagnostics (istep, exit)
    use species, only: nspec, spec, adapt_hc_any, specie
    use theta_grid, only: theta, ntgrid, delthet, jacob
    use theta_grid, only: gradpar, nperiod
    use kgrids, only: naky, nakx, aky, akx, aky
    use kgrids, only: nkpolar 
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use fields, only: phinew, aparnew, bparnew
    use fields, only: phinorm
    use dist_fn, only: flux, write_f
    use dist_fn, only: omega0, gamma0, getmoms, par_spectrum
    use dist_fn, only: get_verr        ! MAB
    use mp, only: proc0, broadcast, iproc
    use file_utils, only: get_unused_unit, flush_output_file
    use agk_time, only: update_time, time
    use agk_io, only: nc_qflux, nc_vflux, nc_pflux, nc_loop, nc_loop_moments
    use agk_io, only: nc_loop_movie
    use agk_layouts, only: yxf_lo
    use agk_transforms, only: init_transforms, transform2
    use le_grids, only: nlambda
    use nonlinear_terms, only: nonlin
    use antenna, only: antenna_w
    use agk_heating, only: heating_diagnostics, init_htype, del_htype, &
         dens_vel_diagnostics,init_dvtype, del_dvtype
! TT>
    use agk_heating, only: zero_htype
! <TT
!>GGH - Needed for adpative hypercollisionality, 2007 SEP 07
    use collisions, only: update_vnewh
!<GGH
    use constants
    implicit none
    integer :: nout = 1
    integer :: nout_movie = 1
    integer, intent (in) :: istep
    logical, intent (out) :: exit
    real, dimension(:,:,:), allocatable :: yxphi, yxapar, yxbpar
    complex, dimension (nakx, naky) :: omega, omegaavg

    type (dens_vel_diagnostics) :: dv
    type (dens_vel_diagnostics), dimension(:,:), allocatable :: dvk
    !GGH J_external
    real, dimension(:,:), allocatable ::  j_ext

    real, dimension (nakx, naky) :: phitot
    complex, dimension (nakx, naky, nspec) :: pfluxneo,qfluxneo
    real :: phi2, apar2, bpar2
    real, dimension (nakx, naky) :: phi2_by_mode, apar2_by_mode, bpar2_by_mode
    real, dimension (nakx, naky, nspec) :: ntot2_by_mode, ntot20_by_mode
    integer :: ig, ik, it, is, unit, il, i, j, nnx, nny, ifield
    complex :: phiavg, sourcefac
    complex, dimension (-ntgrid:ntgrid,nakx,naky,nspec) :: ntot, density, &
         upar, tpar, tperp
    complex, allocatable, dimension (:,:,:) :: phik2
    complex, save :: wtmp_new
    complex :: wtmp_old = 0.
    real, dimension (:), allocatable :: dl_over_b
    real, dimension (2*ntgrid) :: kpar
    real, dimension (nspec) ::  heat_fluxes,  part_fluxes, mom_fluxes,  ntot2, ntot20
    real, dimension (nspec) :: mheat_fluxes, mpart_fluxes, mmom_fluxes
    real, dimension (nspec) :: bheat_fluxes, bpart_fluxes, bmom_fluxes
    real, dimension (nspec) ::  heat_par,  heat_perp
    real, dimension (nspec) :: mheat_par, mheat_perp
    real, dimension (nspec) :: bheat_par, bheat_perp
    real, dimension (naky) :: fluxfac
!    real, dimension (:), allocatable :: phi_by_k, apar_by_k, bpar_by_k
!>MAB
    real, dimension (:,:), allocatable :: errest 
    integer, dimension (:,:), allocatable :: erridx
!<MAB
    real :: hflux_tot, zflux_tot, vflux_tot
    character(200) :: filename
    character(20) :: suffix
    logical :: last = .false.

    exit = .false.

    if (proc0) call get_omegaavg (istep, exit, omegaavg)
    call broadcast (exit)

    if (write_hrate) then
! TT>
       call zero_htype (h)
       call zero_htype (hk)
! <TT
       call heating (istep, h, hk)
    end if

!>GGH
    !Write density and velocity perturbations
    if (write_density_velocity) then
       call init_dvtype (dv,  nspec)   ! memory leak if turned on
       allocate (dvk(nakx, naky))      ! memory leak if turned on
       call init_dvtype (dvk, nspec)   ! memory leak if turned on

       call dens_vel(istep,dv,dvk)
    endif
    !Write Jexternal vs. time
    if (write_jext) then
       allocate (j_ext(nakx, naky)); j_ext=0.
       call calc_jext(istep,j_ext)
    endif
!<GGH

    call update_time

    if (make_movie .and. mod(istep,nmovie)==0) then
       ! EAB 09/17/03 -- modify dump_fields_periodically to print out inverse fft of fields in x,y
       ! write(*,*) "iproc now in dump_fields_periodically case", iproc 
       nnx = yxf_lo%nx
       nny = yxf_lo%ny
       if (use_Phi) then
          allocate (yxphi(nnx,nny,-ntgrid:ntgrid))
          call getmoms (phinew, ntot, density, upar, tpar, tperp)
!          call transform2 (ntot, yxphi, nny, nnx)
          call transform2 (phinew, yxphi, nny, nnx)
       end if
!       if (use_Apar) then
          allocate (yxapar(nnx,nny,-ntgrid:ntgrid))
          call transform2 (ntot, yxapar, nny, nnx)
!          call transform2 (aparnew, yxapar, nny, nnx)
!       end if
       if (use_Bpar) then 
          allocate (yxbpar(nnx,nny,-ntgrid:ntgrid))
          call transform2 (bparnew, yxbpar, nny, nnx)
       end if

       if (proc0) then
          call nc_loop_movie(nout_movie, time, yxphi, yxapar, yxbpar)
       end if

       if (use_Phi) deallocate (yxphi)
!       if (use_Apar) deallocate (yxapar)
       deallocate (yxapar)
       if (use_Bpar) deallocate (yxbpar)
       nout_movie = nout_movie + 1

    end if

! TT>
!    if (mod(istep,nwrite) /= 0 .and. .not. exit) return
    !====================================================!
    ! Go back to the main routine if you don't write out !
    !====================================================!
    if (mod(istep,nwrite) /= 0 .and. .not. exit) then
       !Deallocate variables for density and velocity perturbations
       if (write_density_velocity) then
          call del_dvtype (dv)
          call del_dvtype (dvk)
          deallocate(dvk)
       endif
       ! Deallocate variable for Jexternal
       if (write_jext) deallocate(j_ext)
       return
    end if
    !====================================================!
    ! Go back to the main routine if you don't write out !
    !====================================================!
! <TT

!    if (write_g) call write_f (last)

!>MAB
    if (write_verr) then
       
       allocate(errest(4,2), erridx(4,3))

       errest = 0.0; erridx = 0

       ! error estimate obtained by comparing standard integral with less-accurate integral
       call get_verr (errest, erridx, phinew, bparnew)

       if (proc0 .and. write_ascii) then
          write(res_unit,"(5(1x,e12.6))") time, errest(1,2), errest(2,2), errest(3,2), &
               errest(4,2)
       end if

       deallocate (errest,erridx)
    end if
!<MAB

    if (proc0) then
       omega = omegahist(mod(istep,navg),:,:)
       sourcefac = exp((-zi*omega0+gamma0)*time)
       call phinorm (phitot)
       if (use_Phi)  call get_vol_average (phinew, phinew, phi2, phi2_by_mode)
       if (use_Apar) call get_vol_average (aparnew, aparnew, apar2, apar2_by_mode)
       if (use_Bpar) call get_vol_average (bparnew, bparnew, bpar2, bpar2_by_mode)
    end if

    if (write_any_fluxes) then
       call flux (phinew, aparnew, bparnew, &
             pflux,  qheat,  vflux, &
            pmflux, qmheat, vmflux, &
            pbflux, qbheat, vbflux)
       if (proc0) then
          if (use_Phi) then
             do is = 1, nspec
                qheat(:,:,is,1) = qheat(:,:,is,1)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qheat(:,:,is,1), heat_fluxes(is))
                
                qheat(:,:,is,2) = qheat(:,:,is,2)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qheat(:,:,is,2), heat_par(is))

                qheat(:,:,is,3) = qheat(:,:,is,3)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qheat(:,:,is,3), heat_perp(is))
                
                pflux(:,:,is) = pflux(:,:,is)*spec(is)%dens
                call get_volume_average (pflux(:,:,is), part_fluxes(is))

                vflux(:,:,is) = vflux(:,:,is)*spec(is)%dens*spec(is)%mass*spec(is)%stm
                call get_volume_average (vflux(:,:,is), mom_fluxes(is))

             end do
          end if
          if (use_Apar) then
             do is = 1, nspec
                qmheat(:,:,is,1)=qmheat(:,:,is,1)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qmheat(:,:,is,1), mheat_fluxes(is))

                qmheat(:,:,is,2)=qmheat(:,:,is,2)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qmheat(:,:,is,2), mheat_par(is))

                qmheat(:,:,is,3)=qmheat(:,:,is,3)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qmheat(:,:,is,3), mheat_perp(is))
                
                pmflux(:,:,is)=pmflux(:,:,is)*spec(is)%dens
                call get_volume_average (pmflux(:,:,is), mpart_fluxes(is))

                vmflux(:,:,is)=vmflux(:,:,is)*spec(is)%dens*spec(is)%mass*spec(is)%stm
                call get_volume_average (vmflux(:,:,is), mmom_fluxes(is))
             end do
          end if
          if (use_Bpar) then
             do is = 1, nspec
                qbheat(:,:,is,1)=qbheat(:,:,is,1)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qbheat(:,:,is,1), bheat_fluxes(is))

                qbheat(:,:,is,2)=qbheat(:,:,is,2)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qbheat(:,:,is,2), bheat_par(is))

                qbheat(:,:,is,3)=qbheat(:,:,is,3)*spec(is)%dens*spec(is)%temp
                call get_volume_average (qbheat(:,:,is,3), bheat_perp(is))
                
                pbflux(:,:,is)=pbflux(:,:,is)*spec(is)%dens
                call get_volume_average (pbflux(:,:,is), bpart_fluxes(is))

                vbflux(:,:,is)=vbflux(:,:,is)*spec(is)%dens*spec(is)%mass*spec(is)%stm
                call get_volume_average (vbflux(:,:,is), bmom_fluxes(is))
             end do
          end if
       end if
    end if

    fluxfac = 0.5
    fluxfac(1) = 1.0

    if (proc0) then
       if (print_flux_line) then
          if (use_Phi) then
             write (unit=*, fmt="('t= ',e16.10,' <phi**2>= ',e10.4, &
                  & ' heat fluxes: ', 5(1x,e10.4))") time, phi2, heat_fluxes(1:min(nspec,5))
          end if
          if (use_Apar) then
             write (unit=*, fmt="('t= ',e16.10,' <apar**2>= ',e10.4, &
                  & ' heat flux m: ', 5(1x,e10.4))") time, apar2, mheat_fluxes(1:min(nspec,5))
          end if
          if (use_Bpar) then
             write (unit=*, fmt="('t= ',e16.10,' <bpar**2>= ',e10.4, &
                  & ' heat flux b: ', 5(1x,e10.4))") time, bpar2, bheat_fluxes(1:min(nspec,5))
          end if
       end if
       if (print_line) then
          do ik = 1, naky
             do it = 1, nakx
! TT>
!                write (unit=*, fmt="('ky=',f7.4, ' kx=',f7.4, &
!                     &' om=',2f8.3,' omav=', 2f8.3,' phtot=',e8.2)") &
                write (unit=*, fmt="('ky= ',f8.4, ' kx= ',f8.4, &
                     &' om=',2es15.5,' omav=', 2es15.5,' phtot=',es10.3)") &
! <TT
                     aky(ik), akx(it), &
                     real(omega(it,ik)),    aimag(omega(it,ik)), &
                     real(omegaavg(it,ik)), aimag(omegaavg(it,ik)), &
                     phitot(it,ik)
             end do
          end do
          write (*,*) 
       end if
    end if


!NOTE: THIS IS MOVED TO AFTER POLAR SPECTRUM CALCULATION
    !Call routine to determine adaptive hypercollisionality
    !NOTE: This must have write_hrate=T and come after call to heating
!    if (adapt_hc_any .and. write_hrate .and. proc0) call get_adapt_hc(nuh_changed)
!    call broadcast(nuh_changed)
!    if (nuh_changed) then 
!       call broadcast(spec(1:nspec)%nu_h)
!       !Update arrays in collisions.f90 if nuh_changed=.true.
!       call update_vnewh
!    endif
 
! Polar spectrum calculation----------------------------------------------
    if (write_Epolar .and. proc0) then
       ebinarray(:,:)=0.
       
       !Calculate polar spectrum of energies 
       call get_polar_spectrum (hk%energy, ebinarray(:,iefluc))
       call get_polar_spectrum (hk%eapar,  ebinarray(:,ieapar))
       call get_polar_spectrum (hk%ebpar,  ebinarray(:,iebpar))

       do is=1,nspec
          do ik= 1,naky
             do it=1,nakx
                etmp(it,ik)=hk(it,ik)%phis2(is)
             enddo
          enddo
          call get_polar_spectrum(etmp(:,:), ebinarray(:,iephis2  + (is-1)*3))
          do ik= 1,naky
             do it=1,nakx
                etmp(it,ik)=hk(it,ik)%hs2(is)
             enddo
          enddo
          call get_polar_spectrum(etmp(:,:), ebinarray(:,iehs2    + (is-1)*3))
          do ik= 1,naky
             do it=1,nakx
                etmp(it,ik)=hk(it,ik)%delfs2(is)
             enddo
          enddo
          call get_polar_spectrum(etmp(:,:), ebinarray(:,iedelfs2 + (is-1)*3))
       enddo
       
!
! BD:
! --- must have write_hrate = T for Epolar to work b/c hk array is used

       !Output raw kspectrum to file
       if (nspec == 1) then
          do i=1,nbx
             write (unit=polar_raw_unit, fmt="('t= ',e16.10,' kperp= ',e10.4, &
                  &' E= '     ,e10.4,' Eapar= ',e10.4,' Ebpar= ',e10.4, &
                  &' Ephis2= ',e10.4,' Ehss2= ',e10.4,' Edfs2= ',e10.4)") &
                  & time, kpbin(i),ebinarray(i,1:6)
          end do
       else  !Only writing this data for first two species for now
! Labels assume first species is ion species.
          do i=1,nbx
             write (unit=polar_raw_unit, fmt="('t= ',e16.10,' kperp= ',e10.4, &
                  &' E= '     ,e10.4,' Eapar= ',e10.4,' Ebpar= ',e10.4, &
                  &' Ephii2= ',e10.4,' Ehsi2= ',e10.4,' Edfi2= ',e10.4, &
                  &' Ephie2= ',e10.4,' Ehse2= ',e10.4,' Edfe2= ',e10.4)") &
                  & time, kpbin(i),ebinarray(i,1:9)
          end do
       end if
       write (unit=polar_raw_unit, fmt='(a)') ''      

       !Compute log-averaged polar spectrum
       
       do ifield = 1, 3+nspec*3
          eavgarray(:,ifield)=0.

! GGH: I don't know why these lines were added.  As noticed by TT, they broke the
!      log-average routine
!          if (ifield == 2 .and. use_Apar) then
!             continue
!          else
!             cycle
!          end if

!          if (ifield == 3 .and. use_Apar) then
!             continue
!          else
!             cycle
!          end if

          do i = 1,nbx
             j=polar_avg_index(i)
             eavgarray(j,ifield)=eavgarray(j,ifield)+log(ebinarray(i,ifield))
          enddo
! Finish log-averaging
          do j=1,nkpolar
             eavgarray(j,ifield)=eavgarray(j,ifield)/numavg(j)
          enddo
          eavgarray(:,ifield)=exp(eavgarray(:,ifield))
       end do

! Output log-averaged kspectrum to file
       if (nspec == 1) then
          do i=1,nkpolar
             write (unit=polar_avg_unit, fmt="('t= ',e16.10,' kperp= ',e10.4, &
                  &' E= '     ,e10.4,' Eapar= ',e10.4,' Ebpar= ',e10.4, &
                  &' Ephis2= ',e10.4,' Ehss2= ',e10.4,' Edfs2= ',e10.4)") &
                  & time, kpavg(i),eavgarray(i,1:6)
          end do
       else ! Only writing this data for first two species right now
! Labels assume first species is ion species.
          do i=1,nkpolar
             write (unit=polar_avg_unit, fmt="('t= ',e16.10,' kperp= ',e10.4, &
                  &' E= '     ,e10.4,' Eapar= ',e10.4,' Ebpar= ',e10.4, &
                  &' Ephii2= ',e10.4,' Ehsi2= ',e10.4,' Edfi2= ',e10.4, &
                  &' Ephie2= ',e10.4,' Ehse2= ',e10.4,' Edfe2= ',e10.4)") &
                  & time, kpavg(i),eavgarray(i,1:9)
          end do
       end if
       write (unit=polar_avg_unit, fmt='(a)') ''      

       !FLUSH OUTPUT FILES
       call flush_output_file (polar_raw_unit, ".kspec_raw")
       call flush_output_file (polar_avg_unit, ".kspec_avg")

    end if! END Polar spectrum calculation------------------------------------
! TT>
!    if (.not. (write_any .or. dump_any)) return
    !====================================================!
    ! Go back to the main routine if you don't write out !
    !====================================================!
    if (.not. (write_any .or. dump_any)) then
       !Deallocate variables for density and velocity perturbations
       if (write_density_velocity) then
          call del_dvtype (dv)
          call del_dvtype (dvk)
          deallocate(dvk)
       endif
       ! Deallocate variable for Jexternal
       if (write_jext) deallocate(j_ext)
       return
    end if
    !====================================================!
    ! Go back to the main routine if you don't write out !
    !====================================================!
! <TT

    if (proc0 .and. write_any) then
       if (write_ascii) write (unit=out_unit, fmt=*) 'time=', time
       if (write_ascii .and. write_hrate) then

! TT> added single species case on the right column
! For case with two species:           with single species:
!
! Column     Item
!   1        time                      time                 1
!   2        Energy                    Energy               2
!   3        dEnergy/dt                dEnergy/dt           3
!   4        J_ant.E                   J_ant.E              4
!   5        [h_(i+1)*h_*]/2 * C[h_(i+1)] * T_0, species 1  5
!   6        Col 5, species 2          [h H(h)]_1           6
!   7        [h H(h)]_1                [h C(h)]_1           7
!   8        [h H(h)]_2                [h w_* h]_1          8
!   9        [h C(h)]_1                [chi dh/dt]_1        9
!  10        [h C(h)]_2                sum (h C(h) * T_0)  10
!  11        [h w_* h]_1               sum (h H(h))        11
!  12        [h w_* h]_2               sum (h C(h))        12
!  13        [chi dh/dt]_1             sum (h w_* h)       13
!  14        [chi dh/dt]_2             sum (chi dh/dt)     14
!  15      sum (h C(h) * T_0) like 5   3 + 4 + 13 + 14     15
!  16      sum (h H(h))                (k_perp A)**2       16
!  17      sum (h C(h))                B_par**2            17
!  18      sum (h w_* h)               df_1 ** 2           18
!  19      sum (chi dh/dt)             h_1 ** 2            19
!  20      3 + 4 + 18 + 19             T * (q Phi/T) ** 2  20
!  21      (k_perp A)**2
!  22      B_par**2
!  23      df_1 ** 2
!  24      df_2 ** 2
!  25      h_1 ** 2
!  26      h_2 ** 2
!  27      T * (q Phi/T)_1 ** 2
!  28      T * (q Phi/T)_2 ** 2

          !Output just first two species (ion and electron) into .heat
          if (nspec .ge. 2) then
             write (unit=heat_unit, fmt="(28es12.4)") time,h % energy,  &
                  h % energy_dot, h % antenna, h % imp_colls(1:2), h % hypercoll(1:2), h % collisions(1:2), &
                  h % gradients(1:2), h % heating(1:2), sum(h % imp_colls), sum(h % hypercoll), sum(h % collisions), &
                  sum(h % gradients), sum(h % heating),sum(h%heating)+h%antenna+sum(h%gradients)+h%energy_dot, &
                  h % eapar, h % ebpar, h % delfs2(1:2),  h % hs2(1:2), h % phis2(1:2)
             call flush_output_file (heat_unit, ".heat")
          endif

          do is=1,nspec
!             if (proc0) write(*,'(a,i2,a,i6)')'Output for species ',is,' to unit ',heat_unit2(is)
!             write (*, fmt="(14es12.4)") time
!             write (*, fmt="(14es12.4)") h % energy
!             write (*, fmt="(14es12.4)") h % energy_dot
!             write (*, fmt="(14es12.4)") h % antenna
!             write (*, fmt="(14es12.4)") h % imp_colls(is)
!             write (*, fmt="(14es12.4)") h % hypercoll(is)
!             write (*, fmt="(14es12.4)") h % collisions(is)
!             write (*, fmt="(14es12.4)") h % gradients(is)
!             write (*, fmt="(14es12.4)") h % heating(is)
!             write (*, fmt="(14es12.4)") h % eapar
!             write (*, fmt="(14es12.4)") h % ebpar
!             write (*, fmt="(14es12.4)") h % delfs2(is)
!             write (*, fmt="(14es12.4)") h % hs2(is)
!             write (*, fmt="(14es12.4)") h % phis2(is)
!             write (*, fmt="(i4)") is
!             write (*, fmt="(14es12.4,i4)") time,h % energy,  &
!                  h % energy_dot, h % antenna, h % imp_colls(is), h % hypercoll(is), h % collisions(is), &
!                  h % gradients(is), h % heating(is), &
!                  h % eapar, h % ebpar, h % delfs2(is),  h % hs2(is), h % phis2(is), is
              write (unit=heat_unit2(is), fmt="(14es12.4,i4)") time,h % energy,  &
                  h % energy_dot, h % antenna, h % imp_colls(is), h % hypercoll(is), h % collisions(is), &
                  h % gradients(is), h % heating(is), &
                  h % eapar, h % ebpar, h % delfs2(is),  h % hs2(is), h % phis2(is), is
!             if (proc0) write(*,'(a,i2,a,i6,a)')'Output for species ',is,' to unit ',heat_unit2(is),' succeeded'
             write(suffix,'(a5,i2.2)')".heat",is
!             if (proc0) write(*,'(a,a)')'Flushing file .',suffix
             call flush_output_file (heat_unit2(is),suffix)
!             if (proc0) write(*,'(a,a)')'Flushed file .',suffix
          end do

!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' energy= ',e12.6)") time, h % energy
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' energy_dot= ',e12.6)") time, h % energy_dot
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' J_ant.E= ',e12.6)") time, h % antenna
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' imp_colls= ',12(1x,e12.6))") time, h % imp_colls
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' hyperC= ',12(1x,e12.6))") time, h % hypercoll
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' hCh= ',12(1x,e12.6))") time, h % collisions
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' hw*= ',12(1x,e12.6))") time, h % gradients
!GGH!         write (unit=heat_unit, fmt="('t= ',e12.6,' hwd= ',12(1x,e12.6))") time, h % curvature
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' heating= ',12(1x,e12.6))") time, h % heating

!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_hvisc= ',e12.6)") time, sum(h % hypervisc)
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_hyperC= ',e12.6)") time, sum(h % hypercoll)
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_hCh= ',e12.6)") time, sum(h % collisions)
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_hw*= ',e12.6)") time, sum(h % gradients)
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_heating= ',e12.6)") time, sum(h % heating)

!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_power= ',e12.6)") time, &
!GGH               sum(h%heating)+h%antenna+sum(h%gradients)+h%energy_dot
          !GGH TEST try adding sqrt(2.) to the edot
!GGH          write (unit=heat_unit, fmt="('t= ',e12.6,' total_power= ',e12.6)") time, &
!GGH               sum(h%heating)+h%antenna+sum(h%gradients)+h%energy_dot*sqrt(2.)
!GGH          write (unit=heat_unit, fmt='(a)') ''
       end if

!<GGH
       !Write out data for density and velocity peturbations
       if (write_ascii .and. write_density_velocity) then
          !NOTE: This only works for 2 species
          write (unit=dv_unit, fmt="('t= ',e12.6,' dvpar= ',e12.6,' ', &
               & e12.6,' dvperp= ',e12.6,' ',e12.6,' dn= ',e12.6,' ',e12.6)")  &
               time, dv % dvpar(:), dv % dvperp(:), dv % dn(:)
!          write (unit=dv_unit, fmt="('t= ',e12.6,' dvperp= ',e12.6)") time, dv % dvperp
!          write (unit=dv_unit, fmt="('t= ',e12.6,' dn= ',e12.6)") time, dv % dn
!          write (unit=heat_unit, fmt='(a)') ''
       end if
       !Write out data for j_external
       if (write_ascii .and. write_jext) then
          do ik=1,naky
             do it = 1, nakx
                if (j_ext(ik,it) .ne. 0.) then
                   write (unit=jext_unit, fmt="(es12.4,i4,i4,es12.4)")  &
                        time,it,ik,j_ext(ik,it)
                endif
             enddo
          enddo
       end if
!>GGH
       
       if (use_Apar .and. write_lorentzian .and. write_ascii) then
          wtmp_new = antenna_w()
          if (real(wtmp_old) /= 0. .and. wtmp_new /= wtmp_old) &
               write (unit=out_unit, fmt="('w= ',e16.10, &
               &  ' amp= ',e16.10)") real(wtmp_new), sqrt(2.*apar2)
          wtmp_old = wtmp_new                
       end if

       if (write_nonlin) then
          call nc_loop (nout, time, fluxfac, &
               phinew(igomega,:,:), phi2, phi2_by_mode, &
               aparnew(igomega,:,:), apar2, apar2_by_mode, &
               bparnew(igomega,:,:), bpar2, bpar2_by_mode, &
               h, hk, omega, omegaavg, phitot, write_omega, write_hrate)
          if (write_nl_flux) then
             hflux_tot = 0.
             zflux_tot = 0.
             if (use_Phi) then
                if (write_ascii) then
                   write (unit=out_unit, fmt="('t= ',e16.10,' <phi**2>= ',e10.4, &
                        & ' heat fluxes: ', 5(1x,e10.4))") &
                        time, phi2, heat_fluxes(1:min(nspec,5))
                   write (unit=out_unit, fmt="('t= ',e16.10,' <phi**2>= ',e10.4, &
                        & ' part fluxes: ', 5(1x,e10.4))") &
                        time, phi2, part_fluxes(1:min(nspec,5))
                end if
                hflux_tot = sum(heat_fluxes)
                vflux_tot = sum(mom_fluxes)
                zflux_tot = sum(part_fluxes*spec%z)
             end if
             if (use_Apar) then
                if (write_ascii) then
                   write (unit=out_unit, fmt="('t= ',e16.10,' <apar**2>= ',e10.4, &
                        & ' heat mluxes: ', 5(1x,e10.4))") &
                        time, apar2, mheat_fluxes(1:min(nspec,5))
                   write (unit=out_unit, fmt="('t= ',e16.10,' <apar**2>= ',e10.4, &
                        & ' part mluxes: ', 5(1x,e10.4))") &
                        time, apar2, mpart_fluxes(1:min(nspec,5))
                end if
                hflux_tot = hflux_tot + sum(mheat_fluxes)
                vflux_tot = vflux_tot + sum(mmom_fluxes)
                zflux_tot = zflux_tot + sum(mpart_fluxes*spec%z)
             end if
             if (use_Bpar) then
                if (write_ascii) then
                   write (unit=out_unit, fmt="('t= ',e16.10,' <bpar**2>= ',e10.4, &
                        & ' heat bluxes: ', 5(1x,e10.4))") &
                        time, bpar2, bheat_fluxes(1:min(nspec,5))
                   write (unit=out_unit, fmt="('t= ',e16.10,' <bpar**2>= ',e10.4, &
                        & ' part bluxes: ', 5(1x,e10.4))") &
                        time, bpar2, bpart_fluxes(1:min(nspec,5))
                end if
                hflux_tot = hflux_tot + sum(bheat_fluxes)
                vflux_tot = vflux_tot + sum(bmom_fluxes)
                zflux_tot = zflux_tot + sum(bpart_fluxes*spec%z)
             end if
             if (write_ascii) &
                  write (unit=out_unit, fmt="('t= ',e16.10,' h_tot= ',e10.4, &
                  & ' z_tot= ',e10.4)") time, hflux_tot, zflux_tot
             call nc_qflux (nout, qheat(:,:,:,1), qmheat(:,:,:,1), qbheat(:,:,:,1), &
                  heat_par, mheat_par, bheat_par, &
                  heat_perp, mheat_perp, bheat_perp, &
                  heat_fluxes, mheat_fluxes, bheat_fluxes, hflux_tot)
             call nc_vflux (nout, vflux, vmflux, vbflux, &
                  mom_fluxes, mmom_fluxes, bmom_fluxes, vflux_tot)
             call nc_pflux (nout, pflux, pmflux, pbflux, &
                  part_fluxes, mpart_fluxes, bpart_fluxes, zflux_tot)
          end if
       end if
       if (write_ascii) then
          do ik = 1, naky
             do it = 1, nakx
                if (write_linear) then
! TT>
!                   write (out_unit, "('t= ',e16.10,' aky= ',f5.2, ' akx= ',f5.2, &
!                        &' om= ',2f8.3,' omav= ', 2f8.3,' phtot= ',e8.2)") &
                   write (out_unit, "('t= ',e16.10,' aky= ',f6.2, ' akx= ',f6.2, &
                        &' om= ',2es15.5,' omav= ', 2es15.5,' phtot= ',es10.3)") &
! <TT
                        time, aky(ik), akx(it), &
                        real( omega(it,ik)),    aimag(omega(it,ik)), &
                        real( omegaavg(it,ik)), aimag(omegaavg(it,ik)), &
                        phitot(it,ik)
                end if
                
                if (write_omega) write (out_unit, *) ' omega/(vt/a)=', &
                     real(omega(it,ik)), aimag(omega(it,ik))
                if (write_omavg) write (out_unit, *) ' omavg/(vt/a)=', &
                     real(omegaavg(it,ik)), aimag(omegaavg(it,ik))
             end do
          end do
       end if
    end if

    !Call routine to determine adaptive hypercollisionality
    !NOTE: This must have write_hrate=T and come after call to heating
    if ((adapt_hc_any .or. write_adapt_hc)  .and. write_hrate .and. proc0) call get_adapt_hc(nuh_changed)
    call broadcast(nuh_changed)
    if (nuh_changed) then 
       call broadcast(spec(1:nspec)%nu_h)
       !Update arrays in collisions.f90 if nuh_changed=.true.
       call update_vnewh
    endif

    nout = nout + 1
    if (write_ascii .and. mod(nout, 10) == 0 .and. proc0) &
         call flush_output_file (out_unit, ".out")

!>MAB
    if (write_ascii .and. write_verr .and. mod(nout, 10) == 0 .and. proc0) &
         call flush_output_file (res_unit, ".vres")
!<MAB

!    if (write_hrate) then
!       call del_htype (h)
!       call del_htype (hk)
!       deallocate (hk)
!    end if
!>GGH
    !Deallocate variables for density and velocity perturbations
    if (write_density_velocity) then
       call del_dvtype (dv)
       call del_dvtype (dvk)
       deallocate(dvk)
    endif
! Deallocate variable for Jexternal
    if (write_jext) deallocate(j_ext)
!<GGH


  end subroutine loop_diagnostics

  subroutine heating (istep, h, hk)

    use mp, only: proc0, iproc
    use dist_fn, only: get_heat
    use fields, only: phi, apar, bpar, phinew, aparnew, bparnew
    use species, only: nspec, spec
    use kgrids, only: naky, nakx, aky, akx
    use theta_grid, only: ntgrid, delthet, jacob
    use nonlinear_terms, only: nonlin
    use dist_fn_arrays, only: c_rate
    use agk_heating, only: heating_diagnostics, avg_h, avg_hk
    implicit none
    integer, intent (in) :: istep
    type (heating_diagnostics) :: h
    type (heating_diagnostics), dimension(:,:) :: hk

    real, dimension(-ntgrid:ntgrid) :: wgt
    real :: fac
    integer :: is, ik, it, ig
    
    if (proc0) then
       
       !GGH NOTE: Here wgt is 1/(2*ntgrid+1)
       wgt = delthet*jacob
       wgt = wgt/sum(wgt)
          
       do is = 1, nspec
          do ik = 1, naky
             fac = 0.5
             if (aky(ik) < epsilon(0.)) fac = 1.0
             do it = 1, nakx
                if (aky(ik) < epsilon(0.0) .and. abs(akx(it)) < epsilon(0.0)) cycle
                do ig = -ntgrid, ntgrid
                   
                   !Sum heating by k over all z points (ig)
                   hk(it, ik) % collisions(is) = hk(it, ik) % collisions(is) &
                        + real(c_rate(ig,it,ik,is,1))*fac*wgt(ig)*spec(is)%temp*spec(is)%dens

                   hk(it, ik) % hypercoll(is) = hk(it, ik) % hypercoll(is) &
                        + real(c_rate(ig,it,ik,is,2))*fac*wgt(ig)*spec(is)%temp*spec(is)%dens

                   hk(it, ik) % imp_colls(is) = hk(it, ik) % imp_colls(is) &
                        + real(c_rate(ig,it,ik,is,3))*fac*wgt(ig)*spec(is)%temp*spec(is)%dens

                end do
                h % collisions(is) = h % collisions(is) + hk(it, ik) % collisions(is)
                h % hypercoll(is)  = h % hypercoll(is)  + hk(it, ik) % hypercoll(is)
                h % imp_colls(is)  = h % imp_colls(is)  + hk(it, ik) % imp_colls(is)
             end do
          end do
       end do
    end if

    call get_heat (h, hk, phi, apar, bpar, phinew, aparnew, bparnew)    

    call avg_h(h, h_hist, istep, navg)
    call avg_hk(hk, hk_hist, istep, navg)

  end subroutine heating
!>GGH
!=============================================================================
! Density: Calculate Density perturbations
!=============================================================================
 subroutine dens_vel (istep, dv, dvk)
    use dist_fn, only: get_dens_vel
    use fields, only: phi, apar, bpar, phinew, aparnew, bparnew
    use agk_heating, only: dens_vel_diagnostics, avg_dv, avg_dvk, zero_dvtype
    implicit none
    !Passed
    integer, intent (in) :: istep
    type (dens_vel_diagnostics) :: dv
    type (dens_vel_diagnostics), dimension(:,:) :: dvk

    !Initialize Density and velocity perturbations
    call zero_dvtype(dv)
    call zero_dvtype(dvk)

    !Call routine to calculate density and velocity perturbations
    call get_dens_vel(dv, dvk, phi, apar, bpar, phinew, aparnew, bparnew)    
    
    !Do averages with a history variable
    call avg_dv(dv, dv_hist, istep, navg)
    call avg_dvk(dvk, dvk_hist, istep, navg)

  end subroutine dens_vel
!=============================================================================
! Density: Calculate Density perturbations
!=============================================================================
 subroutine calc_jext (istep, j_ext)
    use mp, only: proc0
    use dist_fn, only: get_jext
    implicit none
    !Passed
    integer, intent (in) :: istep
    real, dimension(:,:) ::  j_ext
    !Local 
    integer :: i

    !Call routine to calculate density and velocity perturbations
    call get_jext(j_ext)    
    
    !Do averages with a history variable
    if (proc0) then
       !Save variable to history
       if (navg > 1) then
          if (istep > 1) &
               j_ext_hist(:,:,mod(istep,navg))= j_ext(:,:)

          !Use average of history
          if (istep >= navg) then
             j_ext=0.
             do i=0,navg-1
                j_ext(:,:) = j_ext(:,:) + j_ext_hist(:,:,i)/ real(navg)
             end do
          end if
       end if
    end if

  end subroutine calc_jext
!=============================================================================
!<GGH

  subroutine get_omegaavg (istep, exit, omegaavg)
    use kgrids, only: naky, nakx
    use fields_arrays, only: phi, apar, bpar, phinew, aparnew, bparnew
    use agk_time, only: dtime
    use constants
    implicit none
    integer, intent (in) :: istep
    logical, intent (in out) :: exit
    complex, dimension (:,:), intent (out) :: omegaavg
    complex, dimension (navg,nakx,naky) :: domega
    integer :: j

    j = igomega
    where (abs(phinew(j,:,:)+aparnew(j,:,:)+bparnew(j,:,:)) < epsilon(0.0) &
           .or. abs(phi(j,:,:)+apar(j,:,:)+bpar(j,:,:)) < epsilon(0.0))
       omegahist(mod(istep,navg),:,:) = 0.0
    elsewhere
       omegahist(mod(istep,navg),:,:) &
            = log((phinew(j,:,:) + aparnew(j,:,:) + bparnew(j,:,:)) &
                  /(phi(j,:,:)   + apar(j,:,:)    + bpar(j,:,:)))*zi/dtime
    end where

    omegaavg = sum(omegahist/real(navg),dim=1)

    if (istep > navg) then
       domega = spread(omegaavg,1,navg) - omegahist
       if (all(sqrt(sum(abs(domega)**2/real(navg),dim=1)) &
            < min(abs(omegaavg),1.0)*omegatol)) &
       then
          if (write_ascii) write (out_unit, "('*** omega converged')")
          exit = .true. .and. exit_when_converged
       end if

       if (any(abs(omegaavg)*dtime > omegatinst)) then
          if (write_ascii) write (out_unit, "('*** numerical instability detected')") 
          exit = .true.
       end if
    end if
  end subroutine get_omegaavg
!================================================================================
! Set up corrections for polar energy spectrum
!================================================================================
!NOTE: Here we calculate the correction factors for each possible kperp
  subroutine init_polar_spectrum
    use dist_fn_arrays, only: kperp2
    use kgrids, only: naky, nakx, aky, akx, nkpolar, ikx, iky
    use constants, only: pi
    use nonlinear_terms, only: nonlin
    use species, only: nspec
    implicit none
!    real, dimension(:,:), allocatable :: kp_by_mode
    integer, dimension(:), allocatable :: num, nbin
    real, dimension(:), allocatable :: kp
    real, dimension(:), allocatable :: kpavg_lim
    real :: kpmax,dkp
    integer :: nkperp                           !Total possible number of kperps
    integer :: ik, it, i,j,inbx !,nkx,nky

! NOTE: In this routine, a square domain is assumed! (nx=ny)
! In read_parameters, write_Epolar => .false. if nx /= ny

    !Determine total number of possible kperps and allocate array
    nkperp = nakx**2 + naky**2
    allocate (num(1:nkperp)); num=0
    allocate (kp(1:nkperp)); kp(:)=huge(kp)
!    allocate(kp_by_mode(nakx,naky)) ; kp_by_mode(:,:)=0.
    allocate (polar_index(nakx,naky)) ; polar_index(:,:)=0

! Loop through all modes and sum number at each kperp
    do it = 1,nakx
       do ik= 1,naky
          if (nonlin .and. it == 1 .and. ik == 1) cycle

! Add to number and calculate magnitude of this kperp
          i = ikx(it)**2 + iky(ik)**2
          num(i)=num(i)+1
          kp(i) = sqrt(kperp2(it, ik))
          polar_index(it,ik)=i
       enddo
    enddo
   
    !Collapse bins to only existing values of kperp
    !Find total number of existing kperps
    nbx=0
    do i=1,nkperp
       if (num(i) > 0) nbx=nbx+1
    enddo
    
    !Allocate bin variables
    allocate (nbin(1:nbx)); nbin=0
    allocate (kpbin(1:nbx)); kpbin=0.
    allocate (ebincorr(1:nbx)); ebincorr=0.
    allocate (ebinarray(1:nbx,3+3*nspec)); ebinarray=0. !Used in loop diagnostics
    allocate (etmp(1:nakx,1:naky)); etmp=0.

    !Copy data
    inbx=0
    do i=1,nkperp
       if (num(i) > 0) then
          inbx=inbx+1
          nbin(inbx)=num(i)
          kpbin(inbx)=kp(i)
          !Correct polar_index
          do it = 1,nakx
             do ik= 1,naky
                if (polar_index(it,ik) == i) polar_index(it,ik)=inbx
             enddo
          enddo
       endif
    enddo

    !Calculate the correction factor for the discrete values
    ebincorr=pi*kpbin/real(nbin)
!ERROR: NOTE This needs to be corrected to look like 
!    ebincorr=pi*(kpbin/kmin)/real(nbin)
! to prevent an accidental division by kmin GGH Jun 07

    !Deallocate variables
    deallocate(num,kp,nbin)

    !Allocate variables for log-averaged polar spectrum
    allocate(kpavg_lim(1:nkpolar+1)); kpavg_lim(:)=0.
    allocate(numavg(1:nkpolar)) ; numavg(:)=0.
    allocate(kpavg(1:nkpolar)) ; kpavg(:)=0.
    allocate(eavgarray(1:nkpolar,9)); eavgarray(:,:)=0. !Used in loop diagnostics
    allocate(polar_avg_index(1:nbx)) ; polar_avg_index(:)=0

    !Determine limits of log-averaged kperp spectra bins
    kpmax=real(int(kpbin(nbx)/kpbin(1))+1)*kpbin(1)
    dkp=(kpmax-kpbin(1))/real(nkpolar)
    do i=1,nkpolar+1
       kpavg_lim(i)=dkp*real(i)
    enddo

    !Do log-average of kperp in each bin and build index
    do i =1,nbx
       do j=1,nkpolar
          if (kpbin(i) .ge. kpavg_lim(j) .and. kpbin(i) .lt. kpavg_lim(j+1) ) then
             numavg(j)=numavg(j)+1.
             kpavg(j)=kpavg(j)+log(kpbin(i))
             polar_avg_index(i)=j
          endif
       enddo
    enddo
    !Finish log-averaging
    kpavg(:)=kpavg(:)/numavg(:)
    kpavg(:)=exp(kpavg(:))

    deallocate(kpavg_lim)

    !Debug
!    if (0. .eq. 0.) then
!       do i=1,nbx
!          write(*,'(i8,es12.4,i8,es12.4)')i, kpbin(i), polar_avg_index(i)
!       enddo
!       write(*,*)'Total kperps= ',nbx,' Total binned= ',sum(numavg)
!       do i=1,nkpolar
!          write(*,'(i8,es12.4,i8)')i, kpavg(i), int(numavg(i))
!       enddo
!    endif

    !Debug
!    if (0. .eq. 1.) then
!       do i=1,nbx
!          write(*,'(i8,es12.4,i8,es12.4)')i, kpbin(i), int(nbin(i)), ebincorr(i)
!       enddo
!       do it = 1,nakx
!          do ik= 1,naky
!             nkx=-(mod((it-1)+int(nakx/2),nakx)-int(nakx/2))
!             nky=ik-1
!            write(*,'(5i8,2es12.4)') it,ik,nkx,nky,polar_index(it,ik),akx(it),aky(ik)
!          enddo
!       enddo
!    endif

  end subroutine init_polar_spectrum
!================================================================================
! Calculate corrected polar spectrum 
!================================================================================
! NOTE: polar_index connects a given mode to the correct binned kperp value
! NOTE: It is assumed here that weighting factors for ky=0 are already 
! incorporated in ee (already done in heating calculations)
  subroutine get_polar_spectrum(ee,ebin)
    use kgrids, only: naky, nakx
    use nonlinear_terms, only: nonlin
    implicit none
    real, dimension (:,:), intent (in) :: ee   !variable ee by (kx,ky) mode
    real, dimension (:), intent (out) :: ebin  !variable ee in kperp bins
    integer :: ik, it

    !Initialize ebin
    ebin=0.

    !Loop through all modes and sum number at each kperp
    do it = 1,nakx
       do ik= 1,naky
          if (nonlin .and. it == 1 .and. ik == 1) cycle
          ebin(polar_index(it,ik))= ebin(polar_index(it,ik))+ee(it,ik)
       enddo
    enddo
    
    !Make corrections for discrete spectrum
    ebin=ebin*ebincorr

  end subroutine get_polar_spectrum
!================================================================================
! Calculate corrected polar spectrum with an average in z
!================================================================================
! NOTE: polar_index connects a given mode to the correct binned kperp value
! NOTE: At the moment, this is unused in the code
  subroutine get_polar_spectrum_zavg(a,b,ebin)
    use theta_grid, only: ntgrid, delthet, jacob
    use kgrids, only: naky, nakx,aky
    use nonlinear_terms, only: nonlin
    implicit none
    real, dimension (-ntgrid:,:,:), intent (in) :: a,b  !data by (ig,kx,ky) mode
    real, dimension (:), intent (out) :: ebin  !variable ee in kperp bins
    integer :: ik, it
    integer :: ng
    real, dimension (-ntgrid:ntgrid) :: wgt
    real :: anorm
    real :: fac                                !Factor for ky=0 modes

    !Get weighting for z-average
    wgt = delthet*jacob
    anorm = sum(wgt)

    !Initialize ebin
    write (*,*) size(ebin),' is size of ebin'
    ebin=0.

    !Loop through all modes and sum number at each kperp
    do it = 1,nakx
       do ik= 1,naky
          fac = 0.5
          if (aky(ik) < epsilon(0.)) fac = 1.0

          if (nonlin .and. it == 1 .and. ik == 1) cycle

          write (*,*) polar_index(it,ik),' should be < nbx?'
          ebin(polar_index(it,ik))= ebin(polar_index(it,ik)) + &
               sum(real(a(:,it,ik)*b(:,it,ik)*wgt))/anorm*fac
       enddo
    enddo
    
    !Make corrections for discrete spectrum
    ebin=ebin*ebincorr

  end subroutine get_polar_spectrum_zavg
!================================================================================
!
!================================================================================
! Deallocate variables used for polar spectrum
  subroutine finish_polar_spectrum
    implicit none

    !Deallocate variables
    deallocate(polar_index,kpbin,ebincorr,ebinarray,etmp)
    deallocate(numavg,kpavg,eavgarray,polar_avg_index)

  end subroutine finish_polar_spectrum
!================================================================================
! Initialize Adaptive Hypercollisionality 
!================================================================================
  subroutine init_adapt_hc
    use dist_fn_arrays, only: kperp2
    use kgrids, only: naky, nakx
    use nonlinear_terms, only: nonlin
    use species, only: nspec,spec,specie
    implicit none
    real :: kpmin,kpmax            !Min/max values of kperp for ahc control
    real :: kp                     !Value of kperp for this (it,ik) mode
    integer :: is,it,ik            !indices
    integer :: ig=0                !z-value

    !Initialize and determine the index for modes within the adpative control
    allocate(ahc_index(1:nspec,1:nakx,1:naky)); ahc_index(:,:,:)=0
    
    !Loop over species and determine which values fall within the desired
    !   kperp band given by kp_hc +/- dkp_hc/2.
    do is=1,nspec
       kpmin=spec(is)%kp_hc-spec(is)%dkp_hc/2.
       kpmax=spec(is)%kp_hc+spec(is)%dkp_hc/2.
       do it = 1,nakx
          do ik= 1,naky
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             kp = sqrt(kperp2(it, ik))
             !Set index to 1 if kperp falls within desired kperp band
             if (kp .ge. kpmin .and. kp .le. kpmax) ahc_index(is,it,ik)=1
             
             !DEBUG
             if (.false. .and. ahc_index(is,it,ik) .eq. 1) then
                write(*,'(a,3i6,a,f6.3)')'(is,it,ik) = ',is,it,ik,' kp= ',kp
             endif

          enddo
       enddo
    enddo

    !Allocate variables
    allocate(pk1(1:nspec),pk2(1:nspec),ek(1:nspec),etot(1:nspec))
    allocate(wnl(1:nspec), kphc(1:nspec),bk(1:nspec))
    allocate(fm1(1:nspec),fm2(1:nspec),ft(1:nspec))
    !Save kperp control values for each species and threshold
    kphc(:)=0.0 ; ft(:)=0.
    do is=1,nspec
       kphc(is)=spec(is)%kp_hc
       ft(is)=spec(is)%gw_hc
    enddo

  end subroutine init_adapt_hc
!================================================================================
! Calculate Diagnostic for Adaptive Hypercollisionality and output results
!================================================================================
! Hypercollisionality adjustment measures, at a given value of kperp (kp_hc) are
! IONS:           fm1= pci+phci / ( E_dfi * wnl ) 
!                 fm2 = phci / ( E_dfi * wnl ) 
! ELECTRONS:      fm1= pci+phci+pce_phce / ( E_tot * wnl ) 
!                 fm2 = phce / ( E_tot * wnl ) 
! To determine if we even try to adjust, we want to ensure significant energy in the
!    modes at this kperp, so we need to compare the local energy with a value
!    over all modes (it,ik).  Thus, 
! IONS:           Sum_k  E_dfi(k)
! ELECTRONS:      Sum_k  E_tot(k)
! The threshold value of gamma/omega is set to ft.  nu_h is changed when
!            abs(fm1-ft)/(fm1+ft) .gt. spec(is)%gw_frac
 subroutine get_adapt_hc(nuh_changed)
    use species, only: nspec,spec,specie
    use agk_heating, only: heating_diagnostics
    use file_utils, only: error_unit, flush_output_file
    use kgrids, only: naky, nakx
    use nonlinear_terms, only: nonlin
    use run_parameters, only: beta
    use agk_time, only: time
    implicit none
    !Passed
    logical, intent(out) :: nuh_changed
    !Local
!    integer, parameter :: maxnspec=128    !Maximum number of species 
!    real, dimension(1:maxnspec) :: pk1,pk2,ek,etot,wnl,kp,nuhold,bk
!    real, dimension(1:maxnspec) :: fm1,fm2,ft !Measures for adjusting nu_h
    integer :: is,it,ik ,is2          !indices
    !Kolmogorov Constants
!    real, parameter :: cm1 = 2.5
!    real, parameter :: cm2 = 2.2
!    real, parameter :: ck1 = 2.5
!    real, parameter :: ck2 = 2.2
    real, parameter :: c2 = 1.1
    integer :: ierr
    real :: tite
    real :: tt
    integer :: nmode
    character(20) :: suffix

    !Initialize variables
    pk1 = 0. ; pk2 = 0. ; ek = 0. ; etot = 0. !; kp = 0. !; nuhold = 0. 
    bk = 0. ; wnl = 0.
    fm1 = 0. ; fm2 = 0. !; ft =0.
    nuh_changed=.false.

    !Loop over all modes (it,ik) for each species to determine heating rates, etc.
    nmode=0
    do is=1,nspec
!       kp(is)=spec(is)%kp_hc
       !Save old value hypercollisionality
!       nuhold(is)=spec(is)%nu_h
!       ft(is)=spec(is)%gw_hc
       !For electron species, we want etot=sum e_tot=h%energy
       if (spec(is)%type .eq. 2)  etot(is)=h%energy   !Total energy
       do it = 1,nakx
          do ik= 1,naky
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             !For ion species, we want etot=sum e_delfi2
             if (spec(is)%type .eq. 1) etot(is)=etot(is)+hk(it,ik)%delfs2(is)
             
             !Skip this mode if not within kperp value for HC control
             if (ahc_index(is,it,ik) .eq. 0) cycle

             !Sum up E_(kperp a_par) 
             bk(is)=bk(is)+hk(it,ik)%eapar
             nmode=nmode+1

             !Choose which values to add depending on species type
             select case(spec(is)%type)
             case(1) !Ions--------------------------------
                !For ions, pk1=pci+phci, pk2 =phci and ek=e_delfi2
                pk1(is)=pk1(is)+hk(it,ik)%collisions(is)+hk(it,ik)%hypercoll(is)
                pk2(is)=pk2(is)+hk(it,ik)%hypercoll(is)
                ek(is)=ek(is)+hk(it,ik)%delfs2(is)
             case(2) !Electrons----------------------
                !For electrons, pk1=pci+phci+pce+phce, pk2=phce and ek=e_all
                do is2=1,nspec
                   pk1(is)=pk1(is)+hk(it,ik)%collisions(is2)+hk(it,ik)%hypercoll(is2)
                enddo
                pk2(is)=pk2(is)+hk(it,ik)%hypercoll(is)
                ek(is)=ek(is)+hk(it,ik)%energy
             case default !Otherwise there is a problem
                ierr=error_unit()
                write(ierr,'(a,i2)')'Error in agk_diagnostics:get_adapt_hc: Unexpected species type ',spec(is)%type
             end select
          enddo
       enddo
    enddo

    !Get bk from E_(kperp a_par)
    !CHECK: Is this correct? It seems to be correct
!ERROR    Here I need to worry about which value of kperp and bk I am using
!NOTE: In normalized units, bk = kperp * Apar/( 2 * sqrt(beta_0))
!    bk(:)=sqrt(bk(:)/real(nmode))  !NOTE: THIS SEEMS TO NOT WORK
    bk(:)=sqrt(kphc(:)*bk(:))
    
    !Calculate wnl from bk(:)
    !GGH NOTE: This assumes ion and electron plasma only with ions as reference
    tite=spec(1)%temp/spec(2)%temp
    wnl(:)=c2*kphc(:)*bk(:)* sqrt(1.+  kphc(:)**2/(beta+2./(1.+1./tite)))

    !Construct measures for adapting collisionality
    fm1(:)=pk1(:)/(ek(:)*wnl(:))
    fm2(:)=pk2(:)/(ek(:)*wnl(:))
    
    !Update hypercollisional coefficients adaptively
    do is=1,nspec
       
       !Only consider adapting nu_h is there is sufficient energy at this kperp
       if (ek(is)/etot(is) .lt. 1.0E-06 .or. .not. spec(is)%adapt_hc) cycle

       if (abs(fm1(is)-ft(is))/(fm1(is)+ft(is)) .gt. spec(is)%gw_frac .and. &
            fm1(is) .gt. ft(is)) then
          !If measured damping is too strong, reduce nuh only 
          !     if Phc (fm2) is dominant and if we are greater than min_nuh
          if (abs(fm2(is)-ft(is))/(fm2(is)+ft(is)) .gt. spec(is)%gw_frac .and. &
               fm2(is) .gt. ft(is) .and. spec(is)%nu_h .gt. spec(is)%min_nuh) then
             spec(is)%nu_h=max( spec(is)%nu_h* (ft(is)/fm2(is)),spec(is)%min_nuh )
!             if (spec(is)%nu_h .lt. spec(is)%min_nuh)  &
!                  spec(is)%nu_h = spec(is)%min_nuh
             nuh_changed=.true.
          endif
       elseif (abs(fm1(is)-ft(is))/(fm1(is)+ft(is)) .gt. spec(is)%gw_frac .and. &
            fm1(is) .lt. ft(is)) then
          !Otherwise, damping is not strong enough, so increase nuh but only 
          !     if we are less than max_nuh     
          if (spec(is)%nu_h .lt. spec(is)%max_nuh) then
             spec(is)%nu_h=min( spec(is)%nu_h* (ft(is)/fm1(is)),spec(is)%max_nuh)
!             if (spec(is)%nu_h .gt. spec(is)%max_nuh)  &
!                  spec(is)%nu_h = spec(is)%max_nuh
             nuh_changed=.true.
          endif
       endif
       
    enddo

    !Output values to ascii file if desired
    tt=time
    if (write_ascii .and. write_adapt_hc) then
       do  is=1,nspec
         write(ahc_unit(is),'(11es12.4)')tt,kphc(is),bk(is),wnl(is), &
              pk1(is),pk2(is),ek(is),etot(is),fm1(is),fm2(is), spec(is)%nu_h
         write(suffix,'(a4,i2.2)')".ahc",is
         suffix=trim(suffix)
         call flush_output_file(ahc_unit(is),suffix)
       enddo
    endif

  end subroutine get_adapt_hc
!================================================================================
! Finalize Adaptive Hypercollisionality 
!================================================================================
  subroutine finish_adapt_hc
    implicit none

    deallocate(ahc_index)
    deallocate(pk1,pk2,ek,etot)
    deallocate(wnl,kphc,bk)
    deallocate(fm1,fm2,ft)

  end subroutine finish_adapt_hc
!================================================================================

!================================================================================
  subroutine get_vol_average_all (a, b, axb, axb_by_mode)
    use theta_grid, only: ntgrid, delthet, jacob
    use kgrids, only: naky, nakx
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: a, b
    real, intent (out) :: axb
    real, dimension (:,:), intent (out) :: axb_by_mode

    integer :: ig, ik, it
    real, dimension (-ntgrid:ntgrid) :: wgt
    real :: anorm

    wgt = delthet*jacob  !NOTE: This is a constant in AstroGK
    anorm = sum(wgt)

    do ik = 1, naky
       do it = 1, nakx
          axb_by_mode(it,ik) = sum(real(conjg(a(:,it,ik))*b(:,it,ik))*wgt)/anorm
       end do
    end do

    call get_volume_average (axb_by_mode, axb)

  end subroutine get_vol_average_all

  subroutine get_vol_average_one (a, b, axb, axb_by_mode)
    use kgrids, only: naky, nakx
    implicit none
    complex, dimension (:,:), intent (in) :: a, b
    real, intent (out) :: axb
    real, dimension (:,:), intent (out) :: axb_by_mode

    integer :: ik, it

    do ik = 1, naky
       do it = 1, nakx
          axb_by_mode(it,ik) = real(conjg(a(it,ik))*b(it,ik))
       end do
    end do

    call get_volume_average (axb_by_mode, axb)
  end subroutine get_vol_average_one

  subroutine get_volume_average (f, favg)
    use mp, only: iproc
    use kgrids, only: naky, nakx, aky
    implicit none
    real, dimension (:,:), intent (in) :: f
    real, intent (out) :: favg
    real :: fac
    integer :: ik, it

! ky=0 modes have correct amplitudes; rest must be scaled
! note contrast with scaling factors in FFT routines.

    favg = 0.
    do ik = 1, naky
       fac = 0.5
       if (aky(ik) == 0.) fac = 1.0
       do it = 1, nakx
          favg = favg + f(it, ik) * fac
       end do
    end do

  end subroutine get_volume_average

end module agk_diagnostics
