module dist_fn
  use init_g, only: ginit
  use redistribute, only: redist_type
  implicit none
  public :: init_dist_fn
  public :: timeadv, exb_shear
  public :: getfieldeq, getan, getmoms
  public :: flux
  public :: ginit, get_epar, get_heat
  public :: t0, omega0, gamma0, source0
  public :: reset_init, write_f, reset_physics
  public :: write_vp !Velocity-space slices for each Fourier mode (GGH 23JAN08)
! TT>
!  public :: M_class, par_spectrum
  public :: par_spectrum, get_init_field
! <TT
  public :: init_kperp2
  public :: get_dens_vel, get_jext !GGH
  public :: get_verr !MAB

  private

  ! knobs
  real, dimension (:), allocatable :: fexp ! (nspec)
  real, dimension (:), allocatable :: bkdiff  ! (nspec)
  integer, dimension (:), allocatable :: bd_exp ! nspec
  real :: poisfac
  real :: t0, omega0, gamma0, thetas, source0
  real :: phi_ext
  real :: aky_star, akx_star
  real :: g_exb

  integer :: adiabatic_option_switch
  integer, parameter :: adiabatic_option_default = 1, &
       adiabatic_option_zero = 2, &
       adiabatic_option_fieldlineavg = 3, &
       adiabatic_option_yavg = 4

  integer :: source_option_switch
  integer, parameter :: source_option_full = 1, &
       source_option_zero = 2, &
       source_option_test1 = 3, source_option_hm_force = 4
  logical :: mult_imp, test, def_parity, even, test_bes
  logical :: accelerated_x = .false.
  logical :: accelerated_v = .false.
  
!! k_parallel filter items
!  real, dimension(:), allocatable :: work, tablekp
!  real :: scale
!  integer :: nwork, ntablekp

  ! internal arrays

  real, dimension (:,:,:), allocatable :: wstar
  ! (naky,negrid,nspec) replicated

  ! fieldeq
  real, dimension (:,:), allocatable :: gamtot, gamtot1, gamtot2, gamtot3
  ! (-ntgrid:ntgrid,nakx,naky) replicated

  complex, dimension (:), allocatable :: a, b, r, ainv
  ! (-ntgrid:ntgrid, -g-layout-)

! TT>
!  complex, dimension (:,:,:), allocatable :: g0, g_h, gnl_1, gnl_2
  complex, dimension (:,:,:), allocatable :: g0, g_h, gnl_1, gnl_2, gnl_3
! <TT
  ! (-ntgrid:ntgrid,2, -g-layout-)

  complex, dimension (:,:), allocatable :: gk0
  
  ! momentum conservation
  real, dimension (:,:,:), allocatable :: sq

  ! linked only
! TT>
!  integer :: M_class
! <TT

  logical :: initialized = .false.
  logical :: initializing = .true.

contains

  subroutine init_dist_fn
    use mp, only: proc0, finish_mp
    use species, only: init_species, nspec
    use theta_grid, only: init_theta_grid, ntgrid
    use kgrids, only: init_kgrids, naky, nakx, akx, aky
    use le_grids, only: init_le_grids, nlambda, negrid
    use run_parameters, only: init_run_parameters
    use collisions, only: init_collisions
    use agk_layouts, only: init_dist_fn_layouts, init_agk_layouts
    use nonlinear_terms, only: init_nonlinear_terms
    use init_g, only: init_init_g
    use hyper, only: init_hyper
    implicit none

    if (initialized) return
    initialized = .true.

    call init_agk_layouts
    call init_species
    call init_theta_grid
    call init_kgrids
    call init_le_grids (accelerated_x, accelerated_v)
    call read_parameters

    if (test) then
       if (proc0) then
          write (*,*) 'nspecies = ',nspec
          write (*,*) 'nlambda = ', nlambda
          write (*,*) 'negrid = ',negrid
          write (*,*) 'nakx = ',nakx
          write (*,*) 'naky = ',naky
       end if
       call finish_mp
       stop
    end if

    call init_run_parameters
    call init_kperp2
    call init_dist_fn_layouts (ntgrid, naky, nakx, nlambda, negrid, nspec)
    if (.not. test_bes) then
       call init_init_g 
       call init_nonlinear_terms 
       call allocate_arrays
    end if
    call init_vpar
    call init_wstar
    call init_bessel

    if (test_bes) then
       call bes_out
       call finish_mp
       stop
    end if

    call init_par_filter
    call init_collisions 
    call init_invert_rhs
    call init_fieldeq
    call init_hyper

  end subroutine init_dist_fn

  subroutine bes_out

    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, ie_idx, is_idx, idx_local, proc_id
    use file_utils, only: open_output_file, close_output_file, get_unused_unit
    use species, only: spec
    use le_grids, only: e, anon
    use dist_fn_arrays, only: aj0, aj1vp2
    use mp, only: proc0, barrier, send, receive

    real :: b0, b1
    integer :: it, ie, il, is, ik, ig, iglo, unit

    if (proc0) then
       call get_unused_unit (unit)
       call open_output_file (unit, ".bes")
    end if


    do iglo=g_lo%llim_world, g_lo%ulim_world
       ik = ik_idx(g_lo, iglo) 
       it = it_idx(g_lo, iglo) 
       is = is_idx(g_lo, iglo) 
       ie = ie_idx(g_lo, iglo) 
       il = il_idx(g_lo, iglo)

       if (idx_local (g_lo, ik, it, il, ie, is)) then
          if (proc0) then 
             b0 = spec(is)%z*anon(ie,is)*aj0(iglo)/spec(is)%temp
             b1 = anon(ie,is)*aj1vp2(iglo)
          else
             call send (spec(is)%z*anon(ie,is)*aj0(iglo)/spec(is)%temp, 0)
             call send (anon(ie,is)*aj1vp2(iglo), 0)
          end if
       else if (proc0) then
          call receive (b0, proc_id(g_lo, iglo))
          call receive (b1, proc_id(g_lo, iglo))
       end if
       
       if (proc0) write (unit, "(i8,1x,3(1x,e12.6))") iglo, b0, b1, e(ie, is)
       
       call barrier
    end do

    
    if (proc0) call close_output_file (unit)


  end subroutine bes_out


  subroutine read_parameters
    use file_utils, only: input_unit, error_unit, input_unit_exist
    use theta_grid, only: nperiod
    use text_options
    use species, only: nspec
    use mp, only: proc0, broadcast
    implicit none
    type (text_option), dimension (5), parameter :: sourceopts = &
         (/ text_option('default', source_option_full), &
            text_option('full', source_option_full), &
            text_option('zero', source_option_zero), &
            text_option('test1', source_option_test1), &
            text_option('hm', source_option_hm_force) /)
    character(20) :: source_option

    type (text_option), dimension (7), parameter :: adiabaticopts = &
         (/ text_option('default', adiabatic_option_default), &
            text_option('no-field-line-average-term', adiabatic_option_default), &
            text_option('field-line-average-term', adiabatic_option_fieldlineavg), &
! eventually add in iphi00 = 0 option:
            text_option('iphi00=0', adiabatic_option_default), &
            text_option('iphi00=1', adiabatic_option_default), &
            text_option('iphi00=2', adiabatic_option_fieldlineavg), &
            text_option('iphi00=3', adiabatic_option_yavg) /)
    character(30) :: adiabatic_option
            
    namelist /dist_fn_knobs/ poisfac, adiabatic_option, mult_imp, test, def_parity, even, &
         g_exb, test_bes
    
    namelist /source_knobs/ t0, omega0, gamma0, source0, &
           phi_ext, source_option, aky_star, akx_star
    integer :: ierr, is, in_file
    logical :: exist
    real :: bd
    logical :: done = .false.

    if (done) return
    done = .true.

    if (proc0) then
       adiabatic_option = 'default'
       poisfac = 0.0
       t0 = 100.0
       source0 = 1.0
       omega0 = 0.0
       gamma0 = 0.0
       aky_star = 0.0
       akx_star = 0.0
       phi_ext = 0.0
       g_exb = 0.0
       mult_imp = .false.
       test = .false.
       test_bes = .false.
       def_parity = .false.
       even = .true.
       source_option = 'default'

       in_file = input_unit_exist("dist_fn_knobs", exist)
       if (exist) read (unit=input_unit("dist_fn_knobs"), nml=dist_fn_knobs)
       in_file = input_unit_exist("source_knobs", exist)
       if (exist) read (unit=input_unit("source_knobs"), nml=source_knobs)

       call get_option_value &
            (source_option, sourceopts, source_option_switch, &
            ierr, "source_option in source_knobs")
       call get_option_value &
            (adiabatic_option, adiabaticopts, adiabatic_option_switch, &
            ierr, "adiabatic_option in dist_fn_knobs")

    end if
    if (.not.allocated(fexp)) allocate (fexp(nspec), bkdiff(nspec), bd_exp(nspec))
    if (proc0) call read_species_knobs

    call broadcast (adiabatic_option_switch)
    call broadcast (poisfac)
    call broadcast (t0)
    call broadcast (source0)
    call broadcast (omega0)
    call broadcast (gamma0)
    call broadcast (aky_star)
    call broadcast (akx_star)
    call broadcast (phi_ext)
    call broadcast (g_exb)
    call broadcast (source_option_switch)
    call broadcast (fexp)
    call broadcast (bkdiff)
    call broadcast (bd_exp)
    call broadcast (mult_imp)
    call broadcast (test)
    call broadcast (test_bes)
    call broadcast (def_parity)
    call broadcast (even)

    if (mult_imp) then
       ! nothing -- fine for linear runs, but not implemented nonlinearly
    else
! consistency check for bkdiff
       bd = bkdiff(1)
       do is = 1, nspec
          if (bkdiff(is) /= bd) then
             if (proc0) write(*,*) 'Forcing bkdiff for species ',is,' equal to ',bd
             if (proc0) write(*,*) 'If this is a linear run, and you want unequal bkdiff'
             if (proc0) write(*,*) 'for different species, specify mult_imp = .true.'
             if (proc0) write(*,*) 'in the dist_fn_knobs namelist.'
             bkdiff(is) = bd
          endif
       end do
    end if

  end subroutine read_parameters 

  subroutine read_species_knobs
    use species, only: nspec
    use file_utils, only: get_indexed_namelist_unit
    implicit none
    integer :: is, unit
    do is = 1, nspec
       fexp(is) = 0.4
       bkdiff(is) = 0.0
       bd_exp(is) = 0
       
       call get_indexed_namelist_unit (unit, "dist_fn_species_knobs", is)
       call fill_species_knobs (unit, fexp(is), bkdiff(is), bd_exp(is))
       close (unit=unit)
    end do
  end subroutine read_species_knobs

  subroutine fill_species_knobs (unit, fexp_out, bakdif_out, bd_exp_out)
    implicit none
    integer, intent (in) :: unit
    real, intent (in out) :: fexp_out
    real, intent (in out) :: bakdif_out
    integer, intent (in out) :: bd_exp_out
    integer :: bd_exp
    real :: fexp, bakdif
    namelist /dist_fn_species_knobs/ fexp, bakdif, bd_exp

    fexp = fexp_out
    bakdif = bakdif_out
    bd_exp = bd_exp_out
    read (unit=unit, nml=dist_fn_species_knobs)
    fexp_out = fexp
    bd_exp_out = bd_exp
    bakdif_out = bakdif
  end subroutine fill_species_knobs


  subroutine init_vpar
    use dist_fn_arrays, only: vpa, vpar, vpac, vperp2
    use species, only: spec
    use theta_grid, only: ntgrid, delthet, gradpar
    use le_grids, only: e, al
    use agk_time, only: dtime
    use agk_layouts, only: g_lo, ik_idx, il_idx, ie_idx, is_idx
    implicit none
    integer :: iglo, is
    real :: al1, e1
    

    if (.not.allocated(vpa)) then
       allocate (vpa    (2,g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (vpac   (2,g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (vperp2 (  g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (vpar   (2,g_lo%llim_proc:g_lo%ulim_alloc))
    endif
    vpa = 0. ; vpac = 0. ; vperp2 = 0. ; vpar = 0.

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       al1 = al(il_idx(g_lo,iglo))
       e1 = e(ie_idx(g_lo,iglo),is_idx(g_lo,iglo))

       vpa(1,iglo) = sqrt(e1*max(0.0, 1.0 - al1))
       vpa(2,iglo) = - vpa(1,iglo)
       vperp2(iglo) = al1*e1

       if (1.0 - al1 < 100.0*epsilon(0.0)) then
          vpa(1,iglo) = 0.0
          vpa(2,iglo) = 0.0
       end if

       if  (1.0 - al1 < 0.0) then
          vpac(1,iglo) = 1.0
          vpac(2,iglo) = -1.0
       else
          vpac(1,iglo) = vpa(1,iglo)
          vpac(2,iglo) = vpa(2,iglo)
       end if

!       vpac(:,iglo) = 0.0

       is = is_idx(g_lo,iglo)
       vpar(1,iglo) = spec(is)%zstm*dtime * abs(gradpar)/delthet * vpac(1,iglo)
       vpar(2,iglo) = spec(is)%zstm*dtime * abs(gradpar)/delthet * vpac(2,iglo)
! ISSUE?  vpar(ntgrid) was set to zero.  Now independent of theta.  Problem anywhere?

    end do

  end subroutine init_vpar

  subroutine init_wstar
    use species, only: nspec, spec
    use kgrids, only: naky, aky
    use le_grids, only: negrid, e
    use agk_time, only: dtime
    implicit none
    integer :: ik, ie, is

    if(.not.allocated(wstar)) allocate (wstar(naky,negrid,nspec))

    do is = 1, nspec
       do ie = 1, negrid
          do ik = 1, naky
! TT> changed sign of wstar
!             wstar(ik,ie,is) = dtime*aky(ik)/2. &
             wstar(ik,ie,is) = -dtime*aky(ik)/2. &
! <TT
                  *(spec(is)%fprim+spec(is)%tprim*(e(ie,is)-1.5))
          end do
       end do
    end do
  end subroutine init_wstar

  subroutine init_bessel
    use dist_fn_arrays, only: aj0, aj1vp2, kperp2, vperp2
    use species, only: spec
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx, aky, akx
    use le_grids, only: e, al
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, ie_idx, is_idx
     implicit none
    integer :: ig, ik, it, il, ie, is
    integer :: iglo
    real :: arg
    logical :: done = .false.

    if (done) return
    done = .true.

    call init_kperp2

    allocate (aj0   (g_lo%llim_proc:g_lo%ulim_alloc))
!    allocate (aj1   (g_lo%llim_proc:g_lo%ulim_alloc))
    allocate (aj1vp2(g_lo%llim_proc:g_lo%ulim_alloc))
    aj0 = 0. ; aj1vp2 = 0.

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       il = il_idx(g_lo,iglo)
       ie = ie_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)

       arg = spec(is)%smz*sqrt(e(ie,is)*al(il)*kperp2(it,ik))  ! |B| = 1 assumed
       aj0(iglo) = j0(arg)
!       aj1(iglo) = j1(arg)
       aj1vp2(iglo) = j1(arg)*vperp2(iglo)*2.0
    end do

  end subroutine init_bessel

  subroutine init_kperp2
    use dist_fn_arrays, only: kperp2
    use species, only: spec
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx, aky, akx
    implicit none
    integer :: ik, it
    logical :: done = .false.

    if (done) return
    done = .true.

    allocate (kperp2(nakx,naky))
    do ik = 1, naky
       do it = 1, nakx
          kperp2(it, ik) = akx(it)**2 + aky(ik)**2
       end do
    end do

  end subroutine init_kperp2

  subroutine init_par_filter
    use theta_grid, only: ntgrid, nperiod
    use agk_transforms, only: init_zf

    call init_zf (ntgrid, nperiod)

  end subroutine init_par_filter

  subroutine par_spectrum(an, an2)

    use agk_transforms, only: kz_spectrum
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx

    complex, dimension(:,:,:) :: an, an2    
    integer :: it, ik
    real :: scale

    call kz_spectrum (an, an2, nakx, naky)
    scale = 1./real(4*ntgrid**2)  
    an2 = an2*scale

  end subroutine par_spectrum

  subroutine init_invert_rhs
    use mp, only: proc0
    use dist_fn_arrays, only: vpa, vpar, vpac
    use species, only: spec
    use theta_grid, only: ntgrid, theta
    use kgrids, only: naky, nakx, aky
    use le_grids, only: negrid, nlambda
    use constants
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, ie_idx, is_idx
    implicit none
    integer :: iglo
    integer :: ig, ik, it, il, ie, is
    real :: vp, bd

    if (.not.allocated(a)) then
       allocate (a   (g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (b   (g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (r   (g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (ainv(g_lo%llim_proc:g_lo%ulim_alloc))
    endif
    a = 0. ; b = 0. ; r = 0. ; ainv = 0.
    
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo,iglo)
       vp = vpar(1,iglo)
       bd = bkdiff(is)

! NOTE: vp was zero for theta=ntgrid in gs2 coding
       ainv(iglo) = 1.0/(1.0 + bd + (1.0-fexp(is))*spec(is)%tz*2.0*vp)
       r(iglo) = (1.0 - bd - (1.0-fexp(is))*spec(is)%tz*2.0*vp)*ainv(iglo)
       a(iglo) = 1.0 + bd - fexp(is)*spec(is)%tz*2.0*vp
       b(iglo) = 1.0 - bd + fexp(is)*spec(is)%tz*2.0*vp
    end do

! TT>
!    M_class = naky*nakx
! <TT

    initializing = .false.

  end subroutine init_invert_rhs

  subroutine allocate_arrays
    use kgrids, only: naky, nakx
    use theta_grid, only: ntgrid
    use dist_fn_arrays, only: g, gnew, gold
    use agk_layouts, only: g_lo
    use nonlinear_terms, only: nonlin
    implicit none
    logical :: alloc = .true.

    if (alloc) then
       allocate (g    (-ntgrid:ntgrid,2,g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (gnew (-ntgrid:ntgrid,2,g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (g0   (-ntgrid:ntgrid,2,g_lo%llim_proc:g_lo%ulim_alloc))
       allocate (gk0                 (2,g_lo%llim_proc:g_lo%ulim_alloc))
       if (nonlin) then
          allocate (gnl_1(-ntgrid:ntgrid,2,g_lo%llim_proc:g_lo%ulim_alloc))
          allocate (gnl_2(-ntgrid:ntgrid,2,g_lo%llim_proc:g_lo%ulim_alloc))
! TT>
          allocate (gnl_3(-ntgrid:ntgrid,2,g_lo%llim_proc:g_lo%ulim_alloc))
!          gnl_1 = 0. ; gnl_2 = 0.
          gnl_1 = 0. ; gnl_2 = 0. ; gnl_3 = 0.
! <TT
       else
! TT>
!          allocate (gnl_1(1,2,1), gnl_2(1,2,1))
          allocate (gnl_1(1,2,1), gnl_2(1,2,1), gnl_3(1,2,1))
! <TT
       end if
    endif

    g = 0. ; gnew = 0. ; g0 = 0. ; gk0 = 0.

    alloc = .false.
  end subroutine allocate_arrays

  subroutine timeadv (phi, apar, bpar, phinew, aparnew, bparnew, istep, mode)

    use theta_grid, only: ntgrid
    use collisions, only: solfp1
    use dist_fn_arrays, only: gnew, g, gold
    use nonlinear_terms, only: add_nonlinear_terms
    use hyper, only: hyper_diff
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in out) :: phi, apar, bpar
    complex, dimension (-ntgrid:,:,:), intent (in out) :: phinew, aparnew, bparnew
    integer, intent (in) :: istep
    integer, optional, intent (in) :: mode
    integer :: modep
    integer :: diagnostics = 1

    modep = 0
    if (present(mode)) modep = mode

    if (modep <= 0) then
! TT>
!       call add_nonlinear_terms (g, gnl_1, gnl_2, &
       call add_nonlinear_terms (gnl_1, gnl_2, gnl_3, &
! <TT
            phi, apar, bpar, istep, bkdiff(1), fexp(1))
       call invert_rhs (phi, apar, bpar, phinew, aparnew, bparnew, istep)
       call hyper_diff (gnew, phinew, bparnew)

       call solfp1 (gnew, g, g0, phinew, bparnew, diagnostics)
       
       if (def_parity) then
          if (even) then
             gnew(-ntgrid:-1, 1,:) = gnew( ntgrid: 1:-1,2,:)
             gnew( 1: ntgrid, 1,:) = gnew(-1:-ntgrid:-1,2,:)
          else
             gnew( 1: ntgrid, 1,:) = -gnew(-1:-ntgrid:-1,2,:)
             gnew(-ntgrid:-1, 1,:) = -gnew( ntgrid: 1:-1,2,:)
          end if
       end if
       
    end if

  end subroutine timeadv

  subroutine exb_shear (g0, phi, apar, bpar)

! TT>
    use mp, only: proc0
! <TT
    use agk_layouts, only: ik_idx, it_idx, g_lo, is_kx_local, idx_local, idx
    use file_utils, only: error_unit
    use theta_grid, only: ntgrid
    use kgrids, only: akx, aky, naky, ikx, nakx, box
    use le_grids, only: negrid, nlambda
    use species, only: nspec
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use dist_fn_arrays, only: kx_shift
    use agk_time, only: dtime

    complex, dimension (-ntgrid:,:,:), intent (in out) :: phi,    apar,    bpar
    complex, dimension (-ntgrid:,:,g_lo%llim_proc:), intent (in out) :: g0
    integer, dimension(:), allocatable, save :: jump, ikx_indexed
    integer, dimension(1), save :: itmin
    integer :: ik, it, ie, is, il, ig, isgn, to_iglo, from_iglo, ierr
    real :: dkx, gdt
! TT> added save attribute for sure
!    logical :: exb_first = .true.
!    logical :: kx_local
    logical, save :: exb_first = .true.
    logical, save :: kx_local
! <TT

! If not in box configuration, return
! TT> added condition for g_exb
!    if (.not. box) return
    if (.not.box .or. abs(g_exb)<epsilon(0.0)) return
! <TT

! If kx data is not local, no ExB shear will be included.

!  THIS IS GGH's fix for this problem---commented out to use TT's
!    call is_kx_local (negrid, nspec, nlambda, naky, nakx, kx_local)
!    if (.not. kx_local) then
!       ierr = error_unit()
!       if (abs(g_exb) > epsilon(0.0)) &
!            write (ierr, &
!            fmt="('Non-zero g_ExB not implemented for this layout.  g_ExB set to zero.')")
!       g_exb = 0.
!       return
!    end if
!=======
! TT> moved below to avoid too many error messages
!    call is_kx_local (negrid, nspec, nlambda, naky, nakx, kx_local)
!    if (.not. kx_local) then
!       ierr = error_unit()
!       g_exb = 0.
!       write (ierr, fmt="('Non-zero g_ExB not implemented for this layout.  g_ExB set to zero.')")
!       return
!    end if
! <TT

! Initialize kx_shift, jump, idx_indexed

    if (exb_first) then
! TT> moved from above to avoid too many error messages
       call is_kx_local (negrid, nspec, nlambda, naky, nakx, kx_local)
       if (.not. kx_local) then
          g_exb = 0.
          if (proc0) then
             ierr = error_unit()
             write (ierr, fmt="('Non-zero g_ExB not implemented for this layout.  g_ExB set to zero.')")
          end if
          return
       end if
! <TT
       exb_first = .false.
       allocate (kx_shift(naky), jump(naky))
       kx_shift = 0.
       jump = 0

       allocate (ikx_indexed(nakx))
       itmin = minloc (ikx)

       do it=itmin(1), nakx
          ikx_indexed (it+1-itmin(1)) = it
       end do

       do it=1,itmin(1)-1
          ikx_indexed (nakx - itmin(1) + 1 + it)= it
       end do
    end if

    dkx = akx(2)
    
! BD: To do: Put the right timestep in here.
!
! For now, approximate Greg's dt == 1/2 (t_(n+1) - t_(n-1))
! with dtime.  
!
! Note: at first time step, there is a difference of a factor of 2.
!

    gdt = dtime

! kx_shift is a function of time.   Update it here:

    do ik=1, naky
       kx_shift(ik) = kx_shift(ik) - aky(ik)*g_exb*gdt
       jump(ik) = nint(kx_shift(ik)/dkx)
       kx_shift(ik) = kx_shift(ik) - jump(ik)*dkx
    end do

! BD: To do: Should save kx_shift array in restart file

    do ik = naky, 2, -1
       if (jump(ik) == 0) exit

       if (jump(ik) < 0) then

          if (use_Phi) then
             do it = 1, nakx + jump(ik)
                do ig=-ntgrid,ntgrid
                   phi (ig,ikx_indexed(it),ik) = phi (ig,ikx_indexed(it-jump(ik)),ik)
                end do
             end do
             do it = nakx + jump(ik) + 1, nakx
                do ig=-ntgrid,ntgrid
                   phi (ig,ikx_indexed(it),ik) = 0.
                end do
             end do
          end if

          if (use_Apar) then
             do it = 1, nakx + jump(ik)
                do ig=-ntgrid,ntgrid
                   apar(ig,ikx_indexed(it),ik) = apar(ig,ikx_indexed(it-jump(ik)),ik)
                end do
             end do
             do it = nakx + jump(ik) + 1, nakx
                do ig=-ntgrid,ntgrid
                   apar (ig,ikx_indexed(it),ik) = 0.
                end do
             end do
          end if

          if (use_Bpar) then
             do it = 1, nakx + jump(ik)
                do ig=-ntgrid,ntgrid
                   bpar(ig,ikx_indexed(it),ik) = bpar(ig,ikx_indexed(it-jump(ik)),ik)
                end do
             end do
             do it = nakx + jump(ik) + 1, nakx
                do ig=-ntgrid,ntgrid
                   bpar (ig,ikx_indexed(it),ik) = 0.
                end do
             end do
          end if

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! For some data layouts, there is no communication required.  Try such a case first.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

          if (kx_local) then
             do is=1,nspec
                do ie=1,negrid
                   do il=1,nlambda
                      do it = 1, nakx + jump(ik)
                         
                         to_iglo = idx(g_lo, ik, ikx_indexed(it), il, ie, is)
                         from_iglo = idx(g_lo, ik, ikx_indexed(it-jump(ik)), il, ie, is)
                         
                         if (idx_local (g_lo, to_iglo)) then
                            do isgn=1,2
                               do ig=-ntgrid,ntgrid
                                  g0(ig,isgn,to_iglo) = g0(ig,isgn,from_iglo)
                               end do
                            end do
                         end if
                      end do

                      do it = nakx + jump(ik) + 1, nakx
                         
                         to_iglo = idx(g_lo, ik, ikx_indexed(it), il, ie, is)
                         
                         if (idx_local (g_lo, to_iglo)) then
                            do isgn=1,2
                               do ig=-ntgrid,ntgrid
                                  g0(ig,isgn,to_iglo) = 0.
                               end do
                            end do
                         end if
                      end do
                   end do
                end do
             end do
          end if

       else  ! case for jump(ik) > 0

          if (use_Phi) then
             do it = nakx, 1+jump(ik), -1
                do ig=-ntgrid,ntgrid
                   phi (ig,ikx_indexed(it),ik) = phi (ig,ikx_indexed(it-jump(ik)),ik)
                end do
             end do
             do it = jump(ik), 1, -1
                do ig=-ntgrid,ntgrid
                   phi (ig,ikx_indexed(it),ik) = 0.
                end do
             end do
          end if

          if (use_Apar) then
             do it = nakx, 1+jump(ik), -1
                do ig=-ntgrid,ntgrid
                   apar(ig,ikx_indexed(it),ik) = apar(ig,ikx_indexed(it-jump(ik)),ik)
                end do
             end do
             do it = jump(ik), 1, -1
                do ig=-ntgrid,ntgrid
                   apar(ig,ikx_indexed(it),ik) = 0.
                end do
             end do
          end if

          if (use_Bpar) then
             do it = nakx, 1+jump(ik), -1
                do ig=-ntgrid,ntgrid
                   bpar(ig,ikx_indexed(it),ik) = bpar(ig,ikx_indexed(it-jump(ik)),ik)
                end do
             end do
             do it = jump(ik), 1, -1
                do ig=-ntgrid,ntgrid
                   bpar(ig,ikx_indexed(it),ik) = 0.
                end do
             end do
          end if

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! For some data layouts, there is no communication required.  Try such a case first.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

          if (kx_local) then
             do is=1,nspec
                do ie=1,negrid
                   do il=1,nlambda
                      do it = nakx, 1+jump(ik), -1

                         to_iglo = idx(g_lo, ik, ikx_indexed(it), il, ie, is)
                         from_iglo = idx(g_lo, ik, ikx_indexed(it-jump(ik)), il, ie, is)

                         if (idx_local (g_lo, to_iglo)) then
                            do isgn=1,2
                               do ig=-ntgrid,ntgrid
                                  g0(ig,isgn,to_iglo) = g0(ig,isgn,from_iglo)
                               end do
                            end do
                         end if
                      end do

                      do it = jump(ik), 1, -1

                         to_iglo = idx(g_lo, ik, ikx_indexed(it), il, ie, is)

                         if (idx_local (g_lo, to_iglo)) then
                            do isgn=1,2
                               do ig=-ntgrid,ntgrid
                                  g0(ig,isgn,to_iglo) = 0.
                               end do
                            end do
                         end if
                      end do
                   end do
                end do
             end do
          end if

       end if

    end do
       
  end subroutine exb_shear

  subroutine g_adjust (g, phi, bpar, facphi, facbpar)
    use species, only: spec
    use theta_grid, only: ntgrid
    use le_grids, only: anon
    use dist_fn_arrays, only: aj0, aj1vp2
    use agk_layouts, only: g_lo, ik_idx, it_idx, ie_idx, is_idx
    implicit none
    complex, dimension (-ntgrid:,:,g_lo%llim_proc:), intent (in out) :: g
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi, bpar
    real, intent (in) :: facphi, facbpar

    integer :: iglo, ig, ik, it, ie, is
    complex :: adj

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)
       ie = ie_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       do ig = -ntgrid, ntgrid
          adj = anon(ie,is)*aj1vp2(iglo)*bpar(ig,it,ik)*facbpar &
               + spec(is)%z*anon(ie,is)*phi(ig,it,ik)*aj0(iglo)/spec(is)%temp*facphi
          g(ig,1,iglo) = g(ig,1,iglo) + adj
          g(ig,2,iglo) = g(ig,2,iglo) + adj
       end do
    end do
  end subroutine g_adjust

  subroutine get_source_term &
       (phi, apar, bpar, phinew, aparnew, bparnew, istep, &
        isgn, iglo, sourcefac, source)
    use dist_fn_arrays, only: aj0, aj1vp2, vpar, vpac, g
    use theta_grid, only: ntgrid, theta
    use kgrids, only: aky, akx
    use le_grids, only: nlambda, anon, e, negrid
    use species, only: spec, nspec
    use run_parameters, only: fphi, fapar, fbpar
    use agk_time, only: dtime
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, ie_idx, is_idx
    use nonlinear_terms, only: nonlin
    use hyper, only: D_res
    use constants
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi,    apar,    bpar
    complex, dimension (-ntgrid:,:,:), intent (in) :: phinew, aparnew, bparnew
    integer, intent (in) :: istep
    integer, intent (in) :: isgn, iglo
    complex, intent (in) :: sourcefac
    complex, dimension (-ntgrid:), intent (out) :: source
    real :: timep

    integer :: ig, ik, it, il, ie, is
    complex, dimension (-ntgrid:ntgrid) :: phigavg, apargavg!, bpargavg !GGH added bpargavg

    ik = ik_idx(g_lo,iglo)
    it = it_idx(g_lo,iglo)
    il = il_idx(g_lo,iglo)
    ie = ie_idx(g_lo,iglo)
    is = is_idx(g_lo,iglo)

    phigavg  = (fexp(is)*phi(:,it,ik) + (1.0-fexp(is))*phinew(:,it,ik)) &
                *aj0(iglo)*fphi &
             + (fexp(is)*bpar(:,it,ik) + (1.0-fexp(is))*bparnew(:,it,ik))&
                *aj1vp2(iglo)*fbpar*spec(is)%tz
    apargavg = (fexp(is)*apar(:,it,ik) + (1.0-fexp(is))*aparnew(:,it,ik)) &
                *aj0(iglo)*fapar

! source term in finite difference equations
    select case (source_option_switch)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Default choice: solve self-consistent equations
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    case (source_option_full)
       call set_source

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Solve self-consistent terms + include external i omega_d Phi * F_0
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    case(source_option_hm_force)
       call set_source
       if (istep > 0 .and. aky(int(aky_star)) == aky(ik) .and. akx(int(akx_star)) == akx(it)) &
            source(:ntgrid-1) = source(:ntgrid-1) - zi*phi_ext*sourcefac

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Include no source term
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    case (source_option_zero)
       source = 0.0

    end select

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Do matrix multiplications
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    if (isgn == 1) then
       do ig = -ntgrid, ntgrid-1
          source(ig) = source(ig) + b(iglo)*g(ig,1,iglo) + a(iglo)*g(ig+1,1,iglo)
       end do
    else
       do ig = -ntgrid, ntgrid-1
          source(ig) = source(ig) + a(iglo)*g(ig,2,iglo) + b(iglo)*g(ig+1,2,iglo)
       end do
    end if

    source(ntgrid) = source(-ntgrid)

  contains

    subroutine set_source

      complex :: apar_p, apar_m, phi_p, phi_m!, bpar_p !GGH added bpar_p
      real, dimension(:), allocatable, save :: ufac
      real :: bd, bdfac_p, bdfac_m
      integer :: i_s
      logical :: first = .true.

      if (first) then
         first = .false.
         allocate (ufac(nspec))
         ufac = 2.0*spec%uprim
      endif

! try fixing bkdiff dependence
      bd = bkdiff(1)

      bdfac_p = 1.+bd*(3.-2.*real(isgn))
      bdfac_m = 1.-bd*(3.-2.*real(isgn))

      do ig = -ntgrid, ntgrid-1
         phi_p = bdfac_p*phigavg(ig+1)+bdfac_m*phigavg(ig)
         phi_m = phigavg(ig+1)-phigavg(ig)
         apar_p = apargavg(ig+1)+apargavg(ig)
         apar_m = aparnew(ig+1,it,ik)+aparnew(ig,it,ik) & 
              -apar(ig+1,it,ik)-apar(ig,it,ik)
 
         source(ig) = anon(ie,is)*(-2.0*vpar(isgn,iglo)*phi_m &
              -spec(is)%zstm*vpac(isgn,iglo) &
              *(aj0(iglo)*apar_m + D_res(it,ik)*apar_p) ) &
              + zi*(wstar(ik,ie,is) &
              + vpac(isgn,iglo)*dtime*aky(ik)/2.0*ufac(is)) &
              *(phi_p - apar_p*spec(is)%stm*vpac(isgn,iglo)) 
      end do
        
      if (nonlin) then         
         select case (istep)
         case (0)
            ! nothing
         case (1)
            do ig = -ntgrid, ntgrid-1
!               source(ig) = source(ig) + 0.5*dtime*gnl_1(ig,isgn,iglo)
! TT: sign changed.  We have to check for the finite bd case...
               source(ig) = source(ig) - 0.5*dtime*gnl_1(ig,isgn,iglo)
            end do
! TT> Updated to AB3
!         case default
         case (2)
! <TT
            do ig = -ntgrid, ntgrid-1
!               source(ig) = source(ig) + 0.5*dtime*( &
! TT: sign changed.  We have to check for the finite bd case...
               source(ig) = source(ig) - 0.5*dtime*( &
                    1.5*gnl_1(ig,isgn,iglo) - 0.5*gnl_2(ig,isgn,iglo))
            end do
! TT>
         case default
            do ig = -ntgrid, ntgrid-1
!               source(ig) = source(ig) + 0.5*code_dt*tfac*( &
!               source(ig) = source(ig) + 0.5*dtime*( &
! TT: sign changed.  We have to check for the finite bd case...
               source(ig) = source(ig) - 0.5*dtime*( &
                    (23./12.) * gnl_1(ig,isgn,iglo) &
                    - (4./3.) * gnl_2(ig,isgn,iglo) &
                    + (5./12.) * gnl_3(ig,isgn,iglo))
            end do
! <TT
         end select
      end if

    end subroutine set_source

  end subroutine get_source_term

  subroutine invert_rhs_1 &
       (phi, apar, bpar, phinew, aparnew, bparnew, istep, &
        iglo, sourcefac)
    use dist_fn_arrays, only: gnew
    use run_parameters, only: eqzip, secondary, tertiary
    use theta_grid, only: ntgrid
    use le_grids, only: nlambda
    use kgrids, only: aky, nakx
    use agk_layouts, only: g_lo, ik_idx, it_idx, il_idx, ie_idx, is_idx
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi,    apar,    bpar
    complex, dimension (-ntgrid:,:,:), intent (in) :: phinew, aparnew, bparnew
    integer, intent (in) :: istep
    integer, intent (in) :: iglo
    complex, intent (in) :: sourcefac

    integer :: ig, ik, it, il, ie, isgn, is
    complex :: ftmp
    complex, dimension (-ntgrid:ntgrid,2) :: source, g1

    ik = ik_idx(g_lo,iglo)
    it = it_idx(g_lo,iglo)
    il = il_idx(g_lo,iglo)
    ie = ie_idx(g_lo,iglo)
    is = is_idx(g_lo,iglo)

    if (eqzip) then
       if (secondary .and. ik == 2 .and. it == 1) return ! do not evolve primary mode
       if (tertiary .and. ik == 1) then
          if (it == 2 .or. it == nakx) return ! do not evolve periodic equilibrium
       end if
    end if

    do isgn = 1, 2
       call get_source_term (phi, apar, bpar, phinew, aparnew, bparnew, &
            istep, isgn, iglo, sourcefac, source(:,isgn))
    end do

    gnew(:,:,iglo) = 0.0              ! gnew is the inhomogeneous solution
    g1 = 0.0                          ! g1 is the homogeneous solution 

    g1(-ntgrid,1) = 1.0               ! initial conditions for g1
    g1( ntgrid,2) = 1.0               ! initial conditions for g1

                                      ! time advance vpar < 0 inhomogeneous part:
    do ig = ntgrid-1, -ntgrid, -1     
       gnew(ig,2,iglo) = -gnew(ig+1,2,iglo)*r(iglo) + ainv(iglo)*source(ig,2)
    end do

                                      ! time advance vpar > 0 inhomogeneous part:
    do ig = -ntgrid, ntgrid-1
       gnew(ig+1,1,iglo) = -gnew(ig,1,iglo)*r(iglo) + ainv(iglo)*source(ig,1)
    end do

    do ig = ntgrid-1, -ntgrid, -1
       g1(ig,2) = -g1(ig+1,2)*r(iglo) ! time advance vpar < 0 homogeneous part:
    end do

    do ig = -ntgrid, ntgrid-1
       g1(ig+1,1) = -g1(ig,1)*r(iglo) ! time advance vpar > 0 homogeneous part: 
    end do
    
    call self_periodic                ! add correct amount of homogeneous solution now

    if (def_parity) then
       if (even) then
          gnew(-ntgrid:-1,1,iglo) = gnew( ntgrid:1:-1,2,iglo)
          gnew(1:ntgrid, 1,iglo) = gnew(-1:-ntgrid:-1,2,iglo)
       else
          gnew(1:ntgrid, 1,iglo) = -gnew(-1:-ntgrid:-1,2,iglo)
          gnew(-ntgrid:-1,1,iglo) = -gnew( ntgrid:1:-1,2,iglo)
       end if
    end if

  contains

    subroutine self_periodic

      if (g1(ntgrid,1) /= 1.) then
         ftmp = (gnew(ntgrid,1,iglo) - gnew(-ntgrid,1,iglo))/(1.0 - g1(ntgrid,1))
         gnew(:,1,iglo) = gnew(:,1,iglo) + ftmp*g1(:,1)
      end if

      if (g1(-ntgrid,2) /= 1.) then
         ftmp = (gnew(-ntgrid,2,iglo) - gnew(ntgrid,2,iglo))/(1.0 - g1(-ntgrid,2))
         gnew(:,2,iglo) = gnew(:,2,iglo) + ftmp*g1(:,2)
      end if
      
    end subroutine self_periodic

  end subroutine invert_rhs_1

  subroutine invert_rhs (phi, apar, bpar, phinew, aparnew, bparnew, istep)
    use dist_fn_arrays, only: gnew
    use theta_grid, only: ntgrid
    use agk_layouts, only: g_lo
    use agk_time, only: time
    use constants
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi,    apar,    bpar
    complex, dimension (-ntgrid:,:,:), intent (in) :: phinew, aparnew, bparnew
    integer, intent (in) :: istep

    integer :: iglo

    real :: timep
    complex :: sourcefac

    if (time > t0) then
       sourcefac = source0*exp(-zi*omega0*time+gamma0*time)
    else
       sourcefac = (0.5 - 0.5*cos(pi*time/t0))*exp(-zi*omega0*time+gamma0*time)
    end if


! This loop is completely parallelizable over the iglo index.
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       call invert_rhs_1 (phi, apar, bpar, phinew, aparnew, bparnew, &
            istep, iglo, sourcefac)
    end do

  end subroutine invert_rhs

  subroutine getan (antot, antota, antotp)
    use dist_fn_arrays, only: vpa, aj0, aj1vp2, gnew
    use species, only: nspec, spec
    use theta_grid, only: ntgrid
    use le_grids, only: integrate_species
    use run_parameters, only: beta, use_Phi, use_Apar, use_Bpar
    use agk_layouts, only: g_lo
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (out) :: antot, antota, antotp
    real, dimension (nspec) :: wgt

    integer :: isgn, iglo, ig

    if (use_Phi) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          do isgn = 1, 2
             do ig=-ntgrid, ntgrid
                g0(ig,isgn,iglo) = aj0(iglo)*gnew(ig,isgn,iglo)
             end do
          end do
       end do

       wgt = spec%z*spec%dens
       call integrate_species (g0, wgt, antot)
    end if

    if (use_Apar) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          do isgn = 1, 2
             do ig=-ntgrid, ntgrid
                g0(ig,isgn,iglo) = aj0(iglo)*vpa(isgn,iglo)*gnew(ig,isgn,iglo)
             end do
          end do
       end do
       
       wgt = 2.0*beta*spec%z*spec%dens*sqrt(spec%temp/spec%mass)
       call integrate_species (g0, wgt, antota)

    end if

    if (use_Bpar) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          do isgn = 1, 2
             do ig=-ntgrid, ntgrid
                g0(ig,isgn,iglo) = 0.5*aj1vp2(iglo)*gnew(ig,isgn,iglo)
             end do
          end do
       end do
       wgt = spec%temp*spec%dens
       call integrate_species (g0, wgt, antotp)

    end if
  end subroutine getan

  subroutine getmoms (phi, ntot, density, upar, tpar, tperp)
    use dist_fn_arrays, only: vpa, vperp2, aj0, gnew
    use agk_layouts, only: is_idx, ie_idx, g_lo, ik_idx, it_idx
    use species, only: nspec, spec
    use theta_grid, only: ntgrid
    use le_grids, only: integrate_moment, anon
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi
    complex, dimension (-ntgrid:,:,:,:), intent (out) :: density, &
         upar, tpar, tperp, ntot

    integer :: ik, it, isgn, ie, is, iglo, ig

! returns moment integrals to PE 0

! total density
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ie = ie_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       ik = ik_idx(g_lo,iglo)
       it = it_idx(g_lo,iglo)

       do isgn = 1, 2
          g0(:,isgn,iglo) = (aj0(iglo)**2-1.0)*anon(ie,is) &
               *phi(:,it,ik)*spec(is)%zt*spec(is)%dens
       end do
    end do

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       do isgn = 1, 2
          do ig=-ntgrid, ntgrid
             g0(ig,isgn,iglo) = aj0(iglo)*gnew(ig,isgn,iglo) + g0(ig,isgn,iglo)
          end do
       end do
    end do
    call integrate_moment (g0, ntot)

    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       do isgn = 1, 2
          do ig=-ntgrid, ntgrid
             g0(ig,isgn,iglo) = gnew(ig,isgn,iglo)
          end do
       end do
    end do

! guiding center density
    call integrate_moment (gnew, density)

! guiding center upar
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       do isgn = 1, 2
          g0(:,isgn,iglo) = vpa(isgn,iglo)*gnew(:,isgn,iglo)
       end do
    end do

    call integrate_moment (g0, upar)

! guiding center tpar
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       do isgn = 1, 2
          g0(:,isgn,iglo) = 2.*vpa(isgn,iglo)*g0(:,isgn,iglo)
       end do
    end do

    call integrate_moment (g0, tpar)
    tpar = tpar - density

! guiding center tperp
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       do isgn = 1, 2
          do ig = -ntgrid, ntgrid
             g0(ig,isgn,iglo) = vperp2(iglo)*gnew(ig,isgn,iglo)
          end do
       end do
    end do

    call integrate_moment (g0, tperp)
    tperp = tperp - density

    do is=1,nspec
       ntot(:,:,:,is)=ntot(:,:,:,is)*spec(is)%dens
       density(:,:,:,is)=density(:,:,:,is)*spec(is)%dens
       upar(:,:,:,is)=upar(:,:,:,is)*spec(is)%stm
       tpar(:,:,:,is)=tpar(:,:,:,is)*spec(is)%temp
       tperp(:,:,:,is)=tperp(:,:,:,is)*spec(is)%temp
    end do

  end subroutine getmoms

  subroutine init_fieldeq
    use dist_fn_arrays, only: aj0, aj1vp2, kperp2
    use species, only: nspec, spec, has_electron_species
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx, aky
    use le_grids, only: anon, integrate_species
    use agk_layouts, only: g_lo, ie_idx, is_idx
    use run_parameters, only: tite
    implicit none
    integer :: iglo, isgn
    integer :: ik, it, ie, is
    complex, dimension (nakx,naky) :: tot
    real, dimension (nspec) :: wgt
    logical :: done = .false.

    if (done) return
    done = .true.

    allocate (gamtot(nakx,naky))
    allocate (gamtot1(nakx,naky))
    allocate (gamtot2(nakx,naky))
    if (adiabatic_option_switch == adiabatic_option_fieldlineavg) allocate (gamtot3(nakx,naky))
    
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ie = ie_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       do isgn = 1, 2
          gk0(isgn,iglo) = (1.0 - aj0(iglo)**2)*anon(ie,is)
       end do
    end do
    wgt = spec%z*spec%z*spec%dens/spec%temp
    call integrate_species (gk0, wgt, tot)
    do ik =1, naky
       do it = 1, nakx
          gamtot(it,ik) = real(tot(it,ik)) + kperp2(it,ik)*poisfac
       end do
    end do
    
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ie = ie_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       do isgn = 1, 2
          gk0(isgn,iglo) = aj0(iglo)*aj1vp2(iglo)*anon(ie,is)
       end do
    end do
    wgt = spec%z*spec%dens
    call integrate_species (gk0, wgt, tot)
    gamtot1 = real(tot)
    
    do iglo = g_lo%llim_proc, g_lo%ulim_proc
       ie = ie_idx(g_lo,iglo)
       is = is_idx(g_lo,iglo)
       do isgn = 1, 2
          gk0(isgn,iglo) = 0.5*aj1vp2(iglo)**2*anon(ie,is)
       end do
    end do
    wgt = spec%temp*spec%dens
    call integrate_species (gk0, wgt, tot)
    gamtot2 = real(tot)

! adiabatic electrons 
    if (.not. has_electron_species(spec)) then
       if (adiabatic_option_switch == adiabatic_option_yavg) then
          do ik = 1, naky
             if (aky(ik) > epsilon(0.0)) gamtot(:,ik) = gamtot(:,ik) + tite
          end do
       elseif (adiabatic_option_switch == adiabatic_option_fieldlineavg) then
          gamtot  = gamtot + tite
          gamtot3 = (gamtot-tite) / gamtot
          where (gamtot3 < 2.*epsilon(0.0)) gamtot3 = 1.0
       else
          gamtot = gamtot + tite 
       endif
    endif
  end subroutine init_fieldeq

  subroutine getfieldeq1 (phi, apar, bpar, antot, antota, antotp, &
       fieldeq, fieldeqa, fieldeqp)
    use dist_fn_arrays, only: kperp2
    use theta_grid, only: ntgrid, delthet, jacob
    use kgrids, only: naky, nakx, aky
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use run_parameters, only: beta, tite
    use species, only: spec, has_electron_species
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi, apar, bpar
    complex, dimension (-ntgrid:,:,:), intent (in) :: antot, antota, antotp
    complex, dimension (-ntgrid:,:,:), intent (out) ::fieldeq,fieldeqa,fieldeqp
    real, allocatable, dimension(:,:), save :: fl_avg, awgt
    integer :: ik, it
    logical :: first = .true.
    
    if (first) allocate (fl_avg(nakx, naky))
    fl_avg = 0.

    if (.not. has_electron_species(spec)) then
       if (adiabatic_option_switch == adiabatic_option_fieldlineavg) then
          
          if (first) then 
             allocate (awgt(nakx, naky))
             awgt = 0.
             do ik = 1, naky
                do it = 1, nakx
                   if (aky(ik) > epsilon(0.0)) cycle
                   awgt(it,ik) = tite/(gamtot(it,ik)*gamtot3(it,ik))
                end do
             end do
          endif
           
          do ik = 1, naky
             do it = 1, nakx
                fl_avg(it,ik) = sum(antot(:,it,ik))*awgt(it,ik)
             end do
          end do

       end if
    end if

    if (use_Phi) then
       do ik = 1, naky
          do it = 1, nakx
             fieldeq(:,it,ik) = antot(:,it,ik) &
                  + bpar(:,it,ik)*gamtot1(it,ik) - gamtot(it,ik)*phi(:,it,ik) 
          end do
       end do
             
       if (.not. has_electron_species(spec)) then
          do ik = 1, naky
             do it = 1, nakx
                fieldeq(:,it,ik) = fieldeq(:,it,ik) + fl_avg(it,ik)
             end do
          end do
       end if
    end if

    if (use_Apar) then
       do ik = 1, naky
          do it = 1, nakx
             fieldeqa(:,it,ik) = antota(:,it,ik) - kperp2(it,ik)*apar(:,it,ik)
          end do
       end do
    end if

    if (use_Bpar) then
       do ik =1, naky
          do it = 1, nakx
             fieldeqp(:,it,ik) = (antotp(:,it,ik) &
                  + bpar(:,it,ik)*gamtot2(it,ik)+0.5*phi(:,it,ik)*gamtot1(it,ik))*beta
          end do
       end do

       do ik = 1, naky
          do it = 1, nakx
             fieldeqp(:,it,ik) = fieldeqp(:,it,ik) + bpar(:,it,ik)
          end do
       end do
    end if

    first = .false.

  end subroutine getfieldeq1

  subroutine getfieldeq (phi, apar, bpar, fieldeq, fieldeqa, fieldeqp)
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi, apar, bpar
    complex, dimension (-ntgrid:,:,:), intent (out) ::fieldeq,fieldeqa,fieldeqp
    complex, dimension (:,:,:), allocatable :: antot, antota, antotp

    allocate (antot (-ntgrid:ntgrid,nakx,naky))
    allocate (antota(-ntgrid:ntgrid,nakx,naky))
    allocate (antotp(-ntgrid:ntgrid,nakx,naky))

    call getan (antot, antota, antotp)
    call getfieldeq1 (phi, apar, bpar, antot, antota, antotp, &
         fieldeq, fieldeqa, fieldeqp)

    deallocate (antot, antota, antotp)
  end subroutine getfieldeq

! TT> Given initial distribution function this obtains consistent fields
  subroutine get_init_field (phi, apar, bpar)
    ! inverts the field equations:
    !   gamtot * phi - gamtot1 * bpar = antot
    !   kperp2 * apar = antota
    !   beta/2 * gamtot1 * phi + (beta * gamtot2 + 1) * bpar = - beta * antotp
    ! I haven't made any check for use_Bpar=T case.
    use run_parameters, only: beta, use_Phi, use_Apar, use_Bpar
    use theta_grid, only: ntgrid
    use kgrids, only: nakx, naky
    use species, only: nspec
    use dist_fn_arrays, only: aj0, vpa, kperp2

    complex, dimension (-ntgrid:,:,:), intent (out) :: phi, apar, bpar
    real, dimension (nspec) :: wgt
    real, dimension (-ntgrid:ntgrid,nakx,naky) :: denominator
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: antot, antota, antotp
    complex, dimension (-ntgrid:ntgrid,nakx,naky) :: numerator

    antot=0.0 ; antota=0.0 ; antotp=0.0
    call getan (antot, antota, antotp)

    ! get phi
    if (use_Phi) then
       numerator = spread(beta * gamtot2 + 1.0, 1, ntgrid*2+1) * antot &
            & - spread(beta * gamtot1, 1, ntgrid*2+1) * antotp
       denominator = spread( (beta * gamtot2 + 1.0) * gamtot &
            & + (beta/2.0) * gamtot1 * gamtot1, 1, ntgrid*2+1)
       where (abs(denominator) < epsilon(0.0)) ! it == ik == 1 only
          phi = 0.0
       elsewhere
          phi = numerator / denominator
       end where
    end if

    ! get apar
    if (use_Apar) then
       denominator = spread(kperp2,1,ntgrid*2+1)
       where (abs(denominator) < epsilon(0.0)) ! it == ik == 1 only
          apar = 0.0
       elsewhere
          apar = antota / denominator
       end where
    end if

    ! get bpar
    if (use_Bpar) then
       numerator = - spread(beta * gamtot, 1, ntgrid*2+1) * antotp &
            & - spread((beta/2.0) * gamtot1, 1, ntgrid*2+1) * antot
       ! following line is actually same with the denom for phi
       denominator = spread(gamtot * (beta * gamtot2 + 1.0) &
            & + (beta/2.0) * gamtot1 * gamtot1, 1, ntgrid*2+1)
       where (abs(denominator) < epsilon(0.0)) ! it == ik == 1 only
          bpar = 0.0
       elsewhere
          bpar = numerator / denominator
       end where
    end if

  end subroutine get_init_field
! <TT

  elemental function j0 (x)
! A&S, p. 369, 9.4
    implicit none
    real, intent (in) :: x
    real :: j0
    real, parameter, dimension (7) :: a = &
         (/ 1.0000000, -2.2499997, 1.2656208, -0.3163866, &
            0.0444479, -0.0039444, 0.0002100 /)
    real, parameter, dimension (7) :: b = &
         (/  0.79788456, -0.00000770, -0.00552740, -0.00009512, &
             0.00137237, -0.00072805,  0.00014476 /)
    real, parameter, dimension (7) :: c = &
         (/ -0.78539816, -0.04166397, -0.00003954,  0.00262573, &
            -0.00054125, -0.00029333,  0.00013558 /)
    real :: y

    if (x <= 3.0) then
       y = (x/3.0)**2
       j0 = a(1)+y*(a(2)+y*(a(3)+y*(a(4)+y*(a(5)+y*(a(6)+y*a(7))))))
    else
       y = 3.0/x
       j0 = (b(1)+y*(b(2)+y*(b(3)+y*(b(4)+y*(b(5)+y*(b(6)+y*b(7))))))) &
            *cos(x+c(1)+y*(c(2)+y*(c(3)+y*(c(4)+y*(c(5)+y*(c(6)+y*c(7))))))) &
            /sqrt(x)
    end if
  end function j0

  elemental function j1 (x)
! A&S, p. 370, 9.4 j1 = 1/x J_1(x)
    implicit none
    real, intent (in) :: x
    real :: j1
    real, parameter, dimension (7) :: a = &
         (/  0.50000000, -0.56249985,  0.21093573, -0.03954289, &
             0.00443319, -0.00031761,  0.00001109 /)
    real, parameter, dimension (7) :: b = &
         (/  0.79788456,  0.00000156,  0.01659667,  0.00017105, &
            -0.00249511,  0.00113653,  0.00020033 /)
    real, parameter, dimension (7) :: c = &
         (/ -2.35619449,  0.12499612,  0.00005650,  -0.00637879, &
             0.00074348,  0.00079824, -0.00029166 /)
    real :: y

    if (x <= 3.0) then
       y = (x/3.0)**2
       j1 = a(1)+y*(a(2)+y*(a(3)+y*(a(4)+y*(a(5)+y*(a(6)+y*a(7))))))
    else
       y = 3.0/x
       j1 = (b(1)+y*(b(2)+y*(b(3)+y*(b(4)+y*(b(5)+y*(b(6)+y*b(7))))))) &
            *cos(x+c(1)+y*(c(2)+y*(c(3)+y*(c(4)+y*(c(5)+y*(c(6)+y*c(7))))))) &
            /x**1.5
    end if
  end function j1

  subroutine flux (phi, apar, bpar, &
        pflux,  qflux,  vflux, &
       pmflux, qmflux, vmflux, &
       pbflux, qbflux, vbflux)
    use species, only: spec
    use theta_grid, only: ntgrid, gradpar
    use kgrids, only: naky, nakx
    use le_grids, only: e
    use dist_fn_arrays, only: g, aj0, vpac, vpa, aj1vp2, vperp2
    use agk_layouts, only: g_lo, ie_idx, is_idx
    use mp, only: proc0
    use run_parameters, only: use_Phi, use_Apar, use_Bpar
    use constants, only: zi
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi, apar, bpar
    real, dimension (:,:,:), intent (out) :: pflux, pmflux, pbflux
    real, dimension (:,:,:), intent (out) :: vflux, vmflux, vbflux
    real, dimension (:,:,:,:), intent (out) :: qflux, qmflux, qbflux
    real :: anorm
    integer :: ig, it, ik, is, isgn
    integer :: iglo

    if (proc0) then
       pflux = 0.0;   qflux = 0.0;   vflux = 0.0
       pmflux = 0.0;  qmflux = 0.0;  vmflux = 0.0
       pbflux = 0.0;  qbflux = 0.0;  vbflux = 0.0
    end if
    
    anorm = sum(cabs(phi) + cabs(apar) + cabs(bpar))
    if (anorm < epsilon(0.0)) return


    if (use_Phi) then
       do isgn = 1, 2
          do ig=-ntgrid,ntgrid
             g0(ig,isgn,:) = g(ig,isgn,:)*aj0
          end do
       end do
       call get_flux (phi, pflux)

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          g0(:,:,iglo) = g0(:,:,iglo)*e(ie_idx(g_lo,iglo), is_idx(g_lo,iglo))
       end do
       call get_flux (phi, qflux(:,:,:,1))

       do isgn = 1, 2
          do ig=-ntgrid,ntgrid
             g0(ig,isgn,:) = g(ig,isgn,:)*2.*vpa(isgn,:)**2*aj0
          end do
       end do
       call get_flux (phi, qflux(:,:,:,2))

       do isgn = 1, 2
          do ig=-ntgrid,ntgrid
             g0(ig,isgn,:) = g(ig,isgn,:)*vperp2*aj0
          end do
       end do
       call get_flux (phi, qflux(:,:,:,3))

       do isgn = 1, 2
          do ig=-ntgrid,ntgrid
             g0(ig,isgn,:) = g(ig,isgn,:)*aj0*vpac(isgn,:)
          end do
       end do
       call get_flux (phi, vflux)

    else
       pflux = 0.
       qflux = 0.
       vflux = 0.
    end if

    if (use_Apar) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) &
                  = -g(:,isgn,iglo)*aj0(iglo)*spec(is)%stm*vpa(isgn,iglo)
          end do
       end do
       call get_flux (apar, pmflux)

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          g0(:,:,iglo) = g0(:,:,iglo)*e(ie_idx(g_lo,iglo), is_idx(g_lo,iglo))
       end do
       call get_flux (apar, qmflux(:,:,:,1))
       
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) &
                  = -g(:,isgn,iglo)*aj0(iglo)*spec(is)%stm*vpa(isgn,iglo) &
                  *2.*vpa(isgn,iglo)**2
          end do
       end do
       call get_flux (apar, qmflux(:,:,:,2))

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) &
                  = -g(:,isgn,iglo)*aj0(iglo)*spec(is)%stm*vpa(isgn,iglo) &
                  *vperp2(iglo)
          end do
       end do
       call get_flux (apar, qmflux(:,:,:,3))
       
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) &
                  = -g(:,isgn,iglo)*aj0(iglo)*spec(is)%stm*vpa(isgn,iglo)*vpac(isgn,iglo)
          end do
       end do
       call get_flux (apar, vmflux)
    else
       pmflux = 0.
       qmflux = 0.
       vmflux = 0.
    end if

    if (use_Bpar) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) = g(:,isgn,iglo)*aj1vp2(iglo)*spec(is)%tz
          end do
       end do
       call get_flux (bpar, pbflux)

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          g0(:,:,iglo) = g0(:,:,iglo)*e(ie_idx(g_lo,iglo), is_idx(g_lo,iglo))
       end do
       call get_flux (bpar, qbflux(:,:,:,1))

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) &
                  = g(:,isgn,iglo)*aj1vp2(iglo)*spec(is)%tz*2.*vpa(isgn,iglo)**2
          end do
       end do
       call get_flux (bpar, qbflux(:,:,:,2))

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) = g(:,isgn,iglo)*aj1vp2(iglo)*spec(is)%tz*vperp2(iglo)
          end do
       end do
       call get_flux (bpar, qbflux(:,:,:,3))

       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo,iglo)
          do isgn = 1, 2
             g0(:,isgn,iglo) = g(:,isgn,iglo)*aj1vp2(iglo)*spec(is)%tz*vpac(isgn,iglo)
          end do
       end do
       call get_flux (bpar, vbflux)
    else
       pbflux = 0.
       qbflux = 0.
       vbflux = 0.
    end if

  end subroutine flux

  subroutine get_flux (fld, flx)
    use theta_grid, only: ntgrid
    use kgrids, only: nakx, aky, naky
    use le_grids, only: integrate_moment
    use species, only: nspec
    use mp, only: proc0
    implicit none
    complex, dimension (-ntgrid:,:,:), intent (in) :: fld
    real, dimension (:,:,:), intent (in out) :: flx
    complex, dimension (:,:,:,:), allocatable :: total
    real :: wgt
    integer :: ik, it, is, ig

    allocate (total(-ntgrid:ntgrid,nakx,naky,nspec))
    call integrate_moment (g0, total)

    if (proc0) then
       do is = 1, nspec
          do ik = 1, naky
             do it = 1, nakx
                flx(it,ik,is) = sum(aimag(total(:,it,ik,is)*conjg(fld(:,it,ik)))) &
                     *aky(ik)
             end do
          end do
       end do

       flx = flx*0.5

    end if

    deallocate (total)
  end subroutine get_flux
!>GGH
!=============================================================================
! Density: Calculate Density perturbations
!=============================================================================
  subroutine get_dens_vel (dv, dvk, phi, apar, bpar, phinew, aparnew, bparnew)
    use constants, only: pi
    use dist_fn_arrays, only: aj0, vpac, vperp2, g, gnew
    use agk_heating, only: dens_vel_diagnostics
    use agk_layouts, only: g_lo, ik_idx, it_idx, is_idx
    use kgrids, only: nakx, naky,aky
    use le_grids, only: integrate_moment
    use mp, only: proc0
    use nonlinear_terms, only: nonlin
    use run_parameters, only: fphi, fbpar
    use species, only: spec,nspec
    use theta_grid, only: jacob, delthet, ntgrid
    implicit none
    !Passed
    type (dens_vel_diagnostics) :: dv
    type (dens_vel_diagnostics), dimension(:,:) :: dvk
    complex, dimension (-ntgrid:,:,:) :: phi, apar, bpar, phinew, aparnew, bparnew
    !Local 
    integer :: isgn, iglo, ig, is, ik, it            !Indices
    complex :: j0phiavg                              !J_0 q phi/T 
    complex :: havg                                  !Time and z centered h
    complex :: phiavg                               !Time and z centered phi
    
    complex, dimension(:,:,:,:), allocatable :: tot  !Integrated DF
    complex, dimension(:,:,:,:), allocatable :: tot2  !Integrated DF
    real :: fac2                                     !Factor
    real, dimension (:), allocatable :: wgt

    !Allocate tot variable for each calculation
    allocate (tot(-ntgrid:ntgrid, nakx, naky, nspec))
!    allocate (tot2(-ntgrid:ntgrid, nakx, naky, nspec))

    !Set up weighting factors for z-sums on proc0
    if (proc0) then
       allocate (wgt(-ntgrid:ntgrid))
       wgt = 0.
       do ig=-ntgrid,ntgrid
          wgt(ig) = delthet*jacob
       end do
       wgt = wgt/sum(wgt)         
    endif

    !CONVERT FROM g TO h
    call g_adjust (g,    phi,    bpar,    fphi, fbpar)
    call g_adjust (gnew, phinew, bparnew, fphi, fbpar)

    !Parallel velocity perturbation--------------------------------------------
    !Loop through and constuct time and z-centered integrands
!    do iglo=g_lo%llim_proc, g_lo%ulim_proc
!       is = is_idx(g_lo, iglo)
!       it = it_idx(g_lo, iglo)
!       ik = ik_idx(g_lo, iglo)
!       if (nonlin .and. it == 1 .and. ik == 1) cycle
!       do isgn=1,2
!          do ig=-ntgrid, ntgrid-1
!
!             !Time and z-centered h
!             havg = favg (g   (ig  ,isgn,iglo), &
!                          g   (ig+1,isgn,iglo), &
!                          gnew(ig  ,isgn,iglo), &
!                          gnew(ig+1,isgn,iglo))
!
!             !NOTE: Do I need vpac here?
!             g0(ig,isgn,iglo) = vpac(isgn,iglo)*havg
!
!          end do
!       end do
!    end do
!
!    !Integrate over velocity
!    call integrate_moment (g0, tot)
!    
!    !Sum over theta (z-direction) to get perpendicular Fourier components only
!    if (proc0) then
!       do ik = 1, naky
!          fac2 = 0.5
!          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
!          do it = 1, nakx
!             !Skip mean value (k=0) components
!             if (nonlin .and. it == 1 .and. ik == 1) cycle
!             do is = 1, nspec
!                do ig = -ntgrid, ntgrid-1
!                   dvk(it,ik)%dvpar(is) = dvk(it,ik)%dvpar(is) &
!                        + real(tot(ig,it,ik,is))*wgt(ig)*fac2
!                end do
!                dv % dvpar(is) = dv%dvpar(is) + dvk(it,ik)%dvpar(is)
!             end do
!          end do
!       end do
!    end if
!
!    !Perpendicular velocity perturbation---------------------------------------
!    !Loop through and constuct time and z-centered integrands
!    !Non-Boltzmann part
!    do iglo=g_lo%llim_proc, g_lo%ulim_proc
!       is = is_idx(g_lo, iglo)
!       it = it_idx(g_lo, iglo)
!       ik = ik_idx(g_lo, iglo)
!       do isgn=1,2
!          do ig=-ntgrid, ntgrid-1
!
!             !Time and z-centered h
!             havg = favg (g   (ig  ,isgn,iglo), &
!                          g   (ig+1,isgn,iglo), &
!                          gnew(ig  ,isgn,iglo), &
!                          gnew(ig+1,isgn,iglo))
!
!             g0(ig,isgn,iglo) = sqrt(vperp2(iglo))*havg
!
!          end do
!       end do
!    end do
!
!    !Integrate over velocity
!    call integrate_moment (g0, tot)
!    
!    !Add in adiabatic part 
!    do iglo=g_lo%llim_proc, g_lo%ulim_proc
!       is = is_idx(g_lo, iglo)
!       it = it_idx(g_lo, iglo)
!       ik = ik_idx(g_lo, iglo)
!       do ig= -ntgrid, ntgrid-1
!          !Time and z-center J_0 Phi
!          j0phi= 0.25*spec(is)%zt* &
!               ( aj0(iglo) *(phi(ig  ,it,ik)+ phinew(ig,  it,ik) ) + &
!                 aj0(iglo) *(phi(ig+1,it,ik)+ phinew(ig+1,it,ik)))
!
!          tot(ig,it,ik,is)=tot(ig,it,ik,is)+ sqrt(pi/2.)*(1.-j0phi)               
!       enddo
!    enddo
!
!    !Sum over theta (z-direction) to get perpendicular Fourier components only
!    if (proc0) then
!       do ik = 1, naky
!          fac2 = 0.5
!          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
!           do it = 1, nakx
!             !Skip mean value (k=0) components
!             if (nonlin .and. it == 1 .and. ik == 1) cycle
!             do is = 1, nspec
!                do ig = -ntgrid, ntgrid-1
!                   dvk(it,ik)%dvperp(is) = dvk(it,ik)%dvperp(is) &
!                        + real(tot(ig,it,ik,is))*wgt(ig)*fac2
!                end do
!                dv % dvperp(is) = dv%dvperp(is) + dvk(it,ik)%dvperp(is)
!             end do
!          end do
!       end do
!    end if
!=============================================================================
!=============================================================================
    !Density perturbation------------------------------------------------------
    !Loop through and constuct time and z-centered integrands
    !Non-Boltzmann part
    do iglo=g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo, iglo)
       it = it_idx(g_lo, iglo)
       ik = ik_idx(g_lo, iglo)
       do isgn=1,2
          do ig=-ntgrid, ntgrid

             havg = gnew(ig,isgn,iglo)

             !Adiabatic part
             phiavg = phinew(ig+1,it,ik) * fphi * spec(is)%zt

             g0(ig,isgn,iglo) = aj0(iglo) * havg - phiavg * spec(is)%dens

          end do
       end do
    end do

    !Integrate over velocity
    call integrate_moment (g0, tot)
!    tot2=0.

    !Calculate Boltzmann part 
!    if (proc0) then
!       do ik=1,naky
!          fac2 = 0.5
!          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
!          do it = 1,nakx
!             if (nonlin .and. it == 1 .and. ik == 1) cycle
!             do is=1,nspec
!                iglo=iglo_idx
!                do ig=-ntgrid, ntgrid-1
!                   
!                   phi_avg = favg (phi   (ig  ,it,ik), &
!                        phi   (ig+1,it,ik), &
!                        phinew(ig  ,it,ik), &
!                        phinew(ig+1,it,ik))
!                   
!                   dvk(it,ik)%dvpar(is) = dvk(it,ik)%dvpar(is) &
!                        + spec(is)%zt*abs(phi_avg)*wgt(ig)*fac2
!
!                end do
!                dv % dvpar(is) = dv%dvpar(is) + dvk(it,ik)%dvpar(is)
!             enddo
!          end do
!       end do
!    endif

    !Sum over theta (z-direction) to get perpendicular Fourier components only
    if (proc0) then
!       dv%dvpar(:)=0.  !Initialize
       do ik = 1, naky
          fac2 = 0.5
          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
          do it = 1, nakx
             !Skip mean value (k=0) components
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             do is = 1, nspec
!                dvk(it,ik)%dvpar(is)=0. !Initialize
                do ig = -ntgrid, ntgrid
!                   dvk(it,ik)%dn(is) = dvk(it,ik)%dn(is) &
!                        + real(tot(ig,it,ik,is))*wgt(ig)*fac2
!                   dvk(it,ik)%dn(is) = dvk(it,ik)%dn(is) &
!                        + abs(tot(ig,it,ik,is))*wgt(ig)*fac2
                   dvk(it,ik)%dn(is) = dvk(it,ik)%dn(is) &
                        + abs(tot(ig,it,ik,is))*wgt(ig)*fac2
!                   dvk(it,ik)%dvpar(is) = dvk(it,ik)%dvpar(is) &
!                        + abs(tot2(ig,it,ik,is))*wgt(ig)*fac2
                end do
                dv % dn(is) = dv%dn(is) + dvk(it,ik)%dn(is)
!                dv % dvpar(is) = dv%dvpar(is) + dvk(it,ik)%dvpar(is)
             end do
          end do
       end do
    end if

    !Calculate Boltzmann part 
!    do iglo=g_lo%llim_proc, g_lo%ulim_proc
!       is = is_idx(g_lo, iglo)
!       it = it_idx(g_lo, iglo)
!       ik = ik_idx(g_lo, iglo)
!       if (nonlin .and. it == 1 .and. ik == 1) cycle
!       do isgn=1,2
!          do ig=-ntgrid, ntgrid-1
!             j0phiavg = favg (phi   (ig  ,it,ik), &
!                  phi   (ig+1,it,ik), &  
!                  phinew(ig  ,it,ik), &  
!                  phinew(ig+1,it,ik)) * aj0(iglo) * fphi * spec(is)%zt
!             
!             g0(ig,isgn,iglo) = -j0phiavg*spec(is)%dens
!
!          end do
!       end do
!    end do
!
!    call integrate_moment (g0, tot)
!
!    !Sum over theta (z-direction) to get perpendicular Fourier components only
!    if (proc0) then
!       do ik = 1, naky
!          fac2 = 0.5
!          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
!          do it = 1, nakx
!             !Skip mean value (k=0) components
!             if (nonlin .and. it == 1 .and. ik == 1) cycle
!             do is = 1, nspec
!                do ig = -ntgrid, ntgrid-1
!                   dvk(it,ik)%dvpar(is) = dvk(it,ik)%dvpar(is) &
!                        + abs(tot(ig,it,ik,is))*wgt(ig)*fac2
!                end do
!                dv % dvpar(is) = dv%dvpar(is) + dvk(it,ik)%dvpar(is)
!             end do
!          end do
!       end do
!    end if
!
    deallocate (tot)
!    deallocate (tot2)

    !CONVERT FROM h BACK TO g
    call g_adjust (g,    phi,    bpar,    -fphi, -fbpar)
    call g_adjust (gnew, phinew, bparnew, -fphi, -fbpar)

!=============================================================================
!=============================================================================
  end subroutine get_dens_vel

!=============================================================================
! Calculate J_external
!=============================================================================
  subroutine get_jext(j_ext)
    use dist_fn_arrays, only: kperp2
    use kgrids, only: nakx, naky,aky
    use mp, only: proc0
    use theta_grid, only: jacob, delthet, ntgrid
    use antenna, only: antenna_apar
    implicit none
    !Passed
    real, dimension(:,:) ::  j_ext
    !Local 
    complex, dimension(:,:,:), allocatable :: j_extz
    integer :: ig,ik, it                             !Indices
    real :: fac2                                     !Factor
    real, dimension (:), allocatable :: wgt
    

    !Get j_ext at current time
    allocate (j_extz(-ntgrid:ntgrid, nakx, naky)) ; j_extz = 0.
    call antenna_apar (kperp2, j_extz)       

    !Set weighting factor for z-averages

    allocate (wgt(-ntgrid:ntgrid))
    !GGH NOTE: Here wgt is 1/(2*ntgrid)
    wgt = 0.
    do ig=-ntgrid,ntgrid
       wgt(ig) = delthet*jacob
    end do
    wgt = wgt/sum(wgt)         
    
    !Take average over z (and weight k modes for later sum?)
    do ig=-ntgrid, ntgrid
       do ik = 1, naky
          fac2 = 0.5
          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
          do it = 1, nakx
             j_ext(it,ik)=j_ext(it,ik)+real(j_extz(ig,it,ik))*wgt(ig)*fac2
          end do
       end do
    enddo

    deallocate (wgt,j_extz)

  end subroutine get_jext
!<GGH
!=============================================================================
  subroutine get_heat (h, hk, phi, apar, bpar, phinew, aparnew, bparnew)
    use mp, only: proc0, iproc
    use constants, only: pi, zi
    use kgrids, only: nakx, naky, aky, akx
    use dist_fn_arrays, only: vpa, vpac, aj0, aj1vp2, g, gnew, kperp2
    use agk_heating, only: heating_diagnostics
    use agk_layouts, only: g_lo, ik_idx, it_idx, is_idx, ie_idx
    use le_grids, only: integrate_moment
    use species, only: spec, nspec,has_electron_species
    use theta_grid, only: jacob, delthet, ntgrid
    use run_parameters, only: fphi, fapar, fbpar, beta, tite
    use agk_time, only: dtime
    use nonlinear_terms, only: nonlin
    use antenna, only: antenna_apar, a_ext_data
    use hyper, only: D_v, D_eta, nexp
    implicit none
    type (heating_diagnostics) :: h
    type (heating_diagnostics), dimension(:,:) :: hk
    complex, dimension (-ntgrid:,:,:) :: phi, apar, bpar, phinew, aparnew, bparnew
    complex, dimension(:,:,:,:), allocatable :: tot
    complex, dimension(:,:,:), allocatable :: epar, bpardot, apardot, phidot, j_ext, a_ext_old, a_ext_new
    complex :: fac, chi, havg, phi_m, apar_m, hdot
    complex :: chidot, j0phiavg, j1bparavg, j0aparavg
    complex :: de, denew, phi_avg
    real, dimension (:), allocatable :: wgt
    real :: fac2, dtinv, akperp4,  akperp2
    real :: bpar2, bpar2new, bperp2, bperp2new, fac3
    integer :: isgn, iglo, ig, is, ik, it, ie
    complex :: phidot1,apardot1,bpardot1
! ==========================================================================
! Ion/Electron heating------------------------------------------------------
! ==========================================================================

    allocate ( phidot(-ntgrid:ntgrid, nakx, naky))
    allocate (apardot(-ntgrid:ntgrid, nakx, naky))
    allocate (bpardot(-ntgrid:ntgrid, nakx, naky))

    call dot ( phi,  phinew,  phidot, fphi)
    call dot (apar, aparnew, apardot, fapar)
    call dot (bpar, bparnew, bpardot, fbpar)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!! Next two calls make the variables g, gnew = h, hnew 
!!! until the end of this procedure!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    call g_adjust (g,    phi,    bpar,    fphi, fbpar)
    call g_adjust (gnew, phinew, bparnew, fphi, fbpar)

    dtinv = 1./dtime
    do iglo=g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo, iglo)
       it = it_idx(g_lo, iglo)
       ik = ik_idx(g_lo, iglo)
       if (nonlin .and. it == 1 .and. ik == 1) cycle
       do isgn=1,2
          
          do ig=-ntgrid, ntgrid
             
             chidot = aj0(iglo)*(phidot(ig,it,ik) - vpac(isgn,iglo) * spec(is)%stm * apardot(ig,it,ik)) &
                  + aj1vp2(iglo)*bpardot(ig,it,ik)*spec(is)%tz
!             phidot1 = fphi* ((1.-fexp(is))*phinew(ig,it,ik)  - fexp(is)*phi(ig,it,ik) )*2.*dtinv
!             apardot1= fapar*((1.-fexp(is))*aparnew(ig,it,ik) - fexp(is)*apar(ig,it,ik))*2.*dtinv
!             bpardot1= fbpar*((1.-fexp(is))*bparnew(ig,it,ik) - fexp(is)*bpar(ig,it,ik))*2.*dtinv
             
             chidot = aj0(iglo)* (  phidot1 - vpac(isgn,iglo) * spec(is)%stm * apardot1) &
                  + aj1vp2(iglo)*bpardot1*spec(is)%tz
             
             hdot = (gnew(ig,isgn,iglo) - g(ig,isgn,iglo))*dtinv
!GGH             hdot = ((1.-fexp(is))*gnew(ig,isgn,iglo) - fexp(is)*g(ig,isgn,iglo))*2.*dtinv
             havg =  gnew(ig,isgn,iglo) + g(ig,isgn,iglo)
!GGH             havg =  (1.-fexp(is))*gnew(ig,isgn,iglo) + fexp(is)*g(ig,isgn,iglo)

! First term on RHS and LHS of Eq B-10 of H1:

             g0(ig,isgn,iglo) = spec(is)%dens*conjg(havg) * (chidot * spec(is)%z - hdot * spec(is)%temp)

          end do
       end do
    end do

    deallocate (phidot, apardot, bpardot)

    allocate (tot(-ntgrid:ntgrid, nakx, naky, nspec))

    call integrate_moment (g0, tot)

    if (proc0) then
       allocate (wgt(-ntgrid:ntgrid))
       wgt = 0.
       do ig=-ntgrid,ntgrid
          wgt(ig) = delthet*jacob
       end do
       wgt = wgt/sum(wgt)

       do is = 1, nspec
          do ik = 1, naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1, nakx
                if (nonlin .and. it == 1 .and. ik == 1) cycle
                do ig = -ntgrid, ntgrid
                    hk(it,ik) % heating(is) = hk(it,ik) % heating(is) + real(tot(ig,it,ik,is))*wgt(ig)*fac2 
                end do
                h % heating(is) = h % heating(is) + hk(it,ik) % heating(is)
             end do
          end do
       end do
    end if

! ==========================================================================
! Antenna Power and B-field contribution to E and E_dot---------------------
! ==========================================================================
    if (proc0) then
       allocate (j_ext(-ntgrid:ntgrid, nakx, naky)) ; j_ext = 0.
       call antenna_apar (kperp2, j_ext)       

       !NOTE: We'll use time decentering from species 1 for the fields 
       !WARNING: THIS MAY BE INCORRECT! GGH
       is=1

       if (beta > epsilon(0.)) then
          dtinv = 1./dtime
          do ik=1,naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1,nakx
                
                if (nonlin .and. it == 1 .and. ik == 1) cycle
                
                do ig=-ntgrid, ntgrid
                   
                   ! J_ext.E when driving antenna only includes A_parallel:
                   apar_m = (aparnew(ig,it,ik)-apar(ig,it,ik)) * dtinv * fapar
!GGH                   apar_m = ((1.-fexp(is))*aparnew(ig,it,ik)-fexp(is)*apar(ig,it,ik)) * 2.*dtinv * fapar
                   hk(it,ik) % antenna = hk(it, ik) % antenna + real(conjg(j_ext(ig,it,ik))*apar_m) * wgt(ig) * fac2

                   bpar2  = real(conjg(bpar(ig,it,ik)) * bpar(ig,it,ik)) * fbpar
                   bperp2 = real(conjg(apar(ig,it,ik)) * apar(ig,it,ik)) * fapar * kperp2(it,ik) * 0.25 ! 0.25 from normalizations

                   bpar2new  = real(conjg(bparnew(ig,it,ik)) * bparnew(ig,it,ik)) * fbpar
                   bperp2new = real(conjg(aparnew(ig,it,ik)) * aparnew(ig,it,ik)) * fapar * kperp2(it,ik) * 0.25 ! 0.25 from normalizations

                   fac3 = wgt(ig) * fac2 * (2.0/beta)

                   hk(it,ik) % energy_dot = hk(it,ik) % energy_dot + 0.5 * (bpar2new + bperp2new - (bpar2 + bperp2)) * dtinv * fac3
!GGH                   hk(it,ik) % energy_dot = hk(it,ik) % energy_dot + 0.5 * ((1.-fexp(is))*(bpar2new + bperp2new) - fexp(is)*(bpar2 + bperp2)) * dtinv * fac3

                   hk(it,ik) % energy = hk(it,ik) % energy + 0.5 * (bpar2new + bperp2new) * fac3

                   !Eapar = int k_perp^2 A_par^2/(8 pi)                   
                   hk(it,ik) % eapar = hk(it,ik) % eapar + 0.5 * bperp2new * fac3 

                   !Ebpar = int B_par^2/(8 pi)
                   hk(it,ik) % ebpar = hk(it,ik) % ebpar + 0.5 * bpar2new * fac3 
                   
                end do
                h % antenna = h % antenna + hk(it, ik) % antenna
                h % eapar = h % eapar + hk(it, ik) % eapar
                h % ebpar = h % ebpar + hk(it, ik) % ebpar
             end do
          end do
       else
          hk % antenna = 0.
          h  % antenna = 0.
          hk % energy_dot = 0.
          hk % energy = 0.
          hk % eapar = 0.
          h  % eapar = 0.
          hk % ebpar = 0.
          h  % ebpar = 0.
       end if
       deallocate (j_ext)
    end if

! ==========================================================================
! Finish E_dot--------------------------------------------------------------
! ==========================================================================

!GGH Include response of Boltzmann species for single-species runs

    if (.not. has_electron_species(spec)) then
       if (proc0) then
          !NOTE: It is assumed here that n0i=n0e and zi=-ze
          dtinv = 1./dtime
          do ik=1,naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1,nakx

                if (nonlin .and. it == 1 .and. ik == 1) cycle

                do ig=-ntgrid, ntgrid

                   !WARNING: THIS IS NOT UPDATED FOR TIME DECENTERING!!!!! GGH
                   phi_m   = (phinew(ig,it,ik) - phi(ig,it,ik))*dtinv

                   !NOTE: Adiabatic (Boltzmann) species has temperature
                   !       T = spec(1)%temp/tite
                   hk(it,ik) % energy_dot = hk(it,ik) % energy_dot + &
                        fphi * real(conjg(phinew(ig,it,ik))*phi_m) &
                        * spec(1)%dens * spec(1)%z * spec(1)%z * (tite/spec(1)%temp) &
                        * wgt(ig)*fac2

                end do
             end do
          end do
       endif
    endif !END Correction to E_dot for single species runs---------------------
 
!GGH New E_dot calc
    dtinv = 1./dtime
    do iglo=g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo, iglo)
       it = it_idx(g_lo, iglo)
       ik = ik_idx(g_lo, iglo)
       if (nonlin .and. it == 1 .and. ik == 1) cycle
       do isgn=1,2

          do ig=-ntgrid, ntgrid
             
             !Calculate old fluctuating energy de
             havg = g(ig,isgn,iglo)

             j0phiavg = phi(ig,it,ik) * aj0(iglo) * fphi * spec(is)%zt

             phi_avg = phi(ig,it,ik) * fphi * spec(is)%zt

             de = 0.5 * spec(is)%temp * spec(is)%dens * &
                  ( conjg(havg) * havg &
                  + conjg(phi_avg) * phi_avg &
                  - conjg(j0phiavg) * havg &
                  - conjg(havg) * j0phiavg )

            !Calculate new fluctuating energy denew
             havg = gnew(ig,isgn,iglo)

             j0phiavg = phinew(ig,it,ik) * aj0(iglo) * fphi * spec(is)%zt

             phi_avg = phinew(ig,it,ik) * fphi * spec(is)%zt

             denew=0.5*spec(is)%temp*spec(is)%dens*(conjg(havg) * havg &
                  + conjg(phi_avg) * phi_avg &
                  - conjg(j0phiavg) * havg &
                  - conjg(havg) * j0phiavg) 

             !Set g0 as the change of energy (denew-de)/dt
             g0(ig,isgn,iglo) = (denew - de) *dtinv
!GGH             g0(ig,isgn,iglo) = ((1.-fexp(is))*denew-fexp(is)*de) * 2.*dtinv

          end do
       end do
    end do
    !GGH -END new e_dot calc

    call integrate_moment (g0, tot)

    if (proc0) then
       do ik = 1, naky
          fac2 = 0.5
          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
          do it = 1, nakx
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             do is = 1, nspec
                do ig = -ntgrid, ntgrid
                   hk(it,ik) % energy_dot = hk(it,ik) % energy_dot + real(tot(ig,it,ik,is)) * wgt(ig) * fac2
                end do
             end do
             h % energy_dot = h % energy_dot + hk(it,ik) % energy_dot
          end do
       end do
    end if


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

! ==========================================================================
! Gradient Contributions to Heating-----------------------------------------
! ==========================================================================

    do iglo=g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo, iglo)
       it = it_idx(g_lo, iglo)
       ik = ik_idx(g_lo, iglo)
       ie = ie_idx(g_lo, iglo)
       if (nonlin .and. it == 1 .and. ik == 1) cycle

       do isgn=1,2
          do ig=-ntgrid, ntgrid

             chi = phinew(ig,it,ik) * aj0(iglo) * fphi &
                  - aparnew(ig,it,ik) * aj0(iglo) * vpac(isgn,iglo) * spec(is)%stm * fapar & 
                  + bparnew(ig,it,ik) * aj1vp2(iglo) * spec(is)%tz * fbpar

             havg = gnew(ig,isgn,iglo)

             g0(ig,isgn,iglo) = zi * wstar(ik,ie,is) * dtinv * conjg(havg) * chi * spec(is)%dens
            
          end do
       end do
    end do

    call integrate_moment (g0, tot)

    if (proc0) then
       do is = 1, nspec
          do ik = 1, naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1, nakx
                if (nonlin .and. it == 1 .and. ik == 1) cycle
                do ig = -ntgrid, ntgrid
                   hk(it,ik) % gradients(is) = hk(it,ik) % gradients(is) + real(tot(ig,it,ik,is)) * wgt(ig) * fac2
                end do
                h % gradients(is) = h % gradients(is) + hk(it,ik) % gradients(is)
             end do
          end do
       end do
    end if
! ==========================================================================
! Hyperviscosity------------------------------------------------------------
! ==========================================================================

    if (D_v > epsilon(0.)) then

       do iglo=g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo, iglo)
          it = it_idx(g_lo, iglo)
          ik = ik_idx(g_lo, iglo)
          if (nonlin .and. it == 1 .and. ik == 1) cycle
          akperp4 = (aky(ik)**2 + akx(it)**2)**nexp
           do isgn=1,2
             do ig=-ntgrid, ntgrid
                
                havg = gnew (ig,isgn,iglo)

                j0phiavg = phinew(ig,it,ik) * aj0(iglo) * fphi * spec(is)%zt

                j1bparavg= bparnew(ig,it,ik) * aj1vp2(iglo) * fbpar

!Set g0 for hyperviscous heating
                g0(ig,isgn,iglo) = spec(is)%dens * spec(is)%temp * D_v * akperp4 * &
                     ( conjg(havg) * havg - conjg(havg) * j0phiavg - conjg(havg) * j1bparavg)

             end do
          end do
       end do

       call integrate_moment (g0, tot)
       if (proc0) then
          do ik = 1, naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1, nakx
                if (nonlin .and. it == 1 .and. ik == 1) cycle
                do is = 1, nspec
                   do ig = -ntgrid, ntgrid
                      hk(it,ik) % hypervisc(is) = hk(it,ik) % hypervisc(is) + real(tot(ig,it,ik,is)) * wgt(ig) * fac2
                   end do
                   h % hypervisc(is) = h % hypervisc(is) + hk(it,ik) % hypervisc(is)
                end do
             end do
          end do
       end if

    end if !End Hyperviscous Heating Calculation


! ==========================================================================
! Hyperresistivity------------------------------------------------------------
! ==========================================================================
 
    if (D_eta > epsilon(0.)) then

       do iglo=g_lo%llim_proc, g_lo%ulim_proc
          is = is_idx(g_lo, iglo)
          it = it_idx(g_lo, iglo)
          ik = ik_idx(g_lo, iglo)
          if (nonlin .and. it == 1 .and. ik == 1) cycle
          akperp4 = (aky(ik)**2 + akx(it)**2)**nexp
           do isgn=1,2
             do ig=-ntgrid, ntgrid
                
                havg = gnew(ig,it,ik)

                j0aparavg = aparnew(ig,it,ik) * aj0(iglo) * fapar * spec(is)%zstm * vpac(isgn,iglo)

!Set g0 for hyperresistive heating
                g0(ig,isgn,iglo) = spec(is)%dens * spec(is)%temp * D_eta * akperp4 * &
                     conjg(havg) * j0aparavg

             end do
          end do
       end do

       call integrate_moment (g0, tot)
       if (proc0) then
          do ik = 1, naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1, nakx
                if (nonlin .and. it == 1 .and. ik == 1) cycle
                do is = 1, nspec
                   do ig = -ntgrid, ntgrid
                      hk(it,ik) % hyperres(is) = hk(it,ik) % hyperres(is) + real(tot(ig,it,ik,is))*wgt(ig)*fac2
                   end do
                   h % hyperres(is) = h % hyperres(is) + hk(it,ik) % hyperres(is)
                end do
             end do
          end do
       end if

    end if !End Hyperresistivity Heating Calculation

!==========================================================================
!Finish Energy-------------------------------------------------------------
!==========================================================================

!GGH Calculate hs2-------------------------------------------------------------
    do iglo=g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo, iglo)
       it = it_idx(g_lo, iglo)
       ik = ik_idx(g_lo, iglo)
       if (nonlin .and. it == 1 .and. ik == 1) cycle
       do isgn=1,2

          do ig=-ntgrid, ntgrid
             
             havg = gnew(ig,isgn,iglo)

             g0(ig,isgn,iglo) = 0.5 * spec(is)%temp * spec(is)%dens * (conjg(havg) * havg)
          end do
       end do
    end do

    call integrate_moment (g0, tot)

    if (proc0) then
       do ik = 1, naky
          fac2 = 0.5
          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
          do it = 1, nakx
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             do is = 1, nspec             
                do ig = -ntgrid, ntgrid

                   !hs2 = int_r int_v T/F0 hs^2/2
                   hk(it,ik) % hs2(is) = hk(it,ik) % hs2(is) + real(tot(ig,it,ik,is)) * wgt(ig) * fac2

                end do
             end do
             h % hs2(:) = h % hs2(:) + hk(it,ik) % hs2(:)
          end do
       end do
    end if

!Calculate phis2-------------------------------------------------------------
    if (proc0) then
       do ik=1,naky
          fac2 = 0.5
          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
          do it = 1,nakx
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             do ig=-ntgrid, ntgrid
                do is = 1, nspec
                   phi_avg = phinew(ig,it,ik) * fphi * spec(is)%zt

                   !hs2 = int_r int_v T/F0 hs^2/2
                   hk(it,ik) % phis2(is) = hk(it,ik) % phis2(is)  &
                        + 0.5 * spec(is)%temp * spec(is)%dens * real(conjg(phi_avg) * phi_avg) &
                        * wgt(ig) * fac2
                enddo
             end do
             h % phis2(:) = h % phis2(:) + hk(it,ik) % phis2(:)
          end do
       end do
    endif

! Calculate delfs2 (rest of energy)-----------------------------------------------

!GGH  Include response of Boltzmann species for single species runs
    if (.not. has_electron_species(spec)) then
       dtinv = 1./dtime
       if (proc0) then
          !NOTE: It is assumed here that n0i=n0e and zi=-ze
          do ik=1,naky
             fac2 = 0.5
             if (aky(ik) < epsilon(0.0)) fac2 = 1.0
             do it = 1,nakx

                if (nonlin .and. it == 1 .and. ik == 1) cycle

                do ig=-ntgrid, ntgrid

                   phi_avg = phinew(ig,it,ik) * fphi

                   !NOTE: Adiabatic (Boltzmann) species has temperature
                   !       T = spec(1)%temp/tite
                   hk(it,ik) % energy = hk(it,ik) % energy + &
                        fphi * real(conjg(phi_avg) * phi_avg) &
                        * 0.5 * spec(1)%dens * spec(1)%z * spec(1)%z * (tite/spec(1)%temp) &
                        * wgt(ig)*fac2

                end do
             end do
          end do
       endif
    endif !END Correction to energy for single species runs---------------------

    do iglo=g_lo%llim_proc, g_lo%ulim_proc
       is = is_idx(g_lo, iglo)
       it = it_idx(g_lo, iglo)
       ik = ik_idx(g_lo, iglo)
       if (nonlin .and. it == 1 .and. ik == 1) cycle
       do isgn=1,2

          do ig=-ntgrid, ntgrid
             
             havg = gnew(ig,isgn,iglo)

             j0phiavg = phinew(ig,it,ik) * aj0(iglo) * fphi * spec(is)%zt

             phi_avg =  phinew(ig,it,ik) * fphi * spec(is)%zt

             g0(ig,isgn,iglo) = 0.5 * spec(is)%temp * spec(is)%dens * (conjg(havg) * havg &
                  + conjg(phi_avg) * phi_avg &
                  - conjg(j0phiavg) * havg &
                  - conjg(havg) * j0phiavg)
          end do
       end do
    end do

    call integrate_moment (g0, tot)

    if (proc0) then
       do ik = 1, naky
          fac2 = 0.5
          if (aky(ik) < epsilon(0.0)) fac2 = 1.0
          do it = 1, nakx
             if (nonlin .and. it == 1 .and. ik == 1) cycle
             do is = 1, nspec             
                do ig = -ntgrid, ntgrid
                   hk(it,ik) % energy = hk(it,ik) % energy + real(tot(ig,it,ik,is)) * wgt(ig) * fac2

                   !Delfs2 = int_r int_v T/F0 dfs^2/2
                   hk(it,ik) % delfs2(is) = hk(it,ik) % delfs2(is) + real(tot(ig,it,ik,is)) * wgt(ig) * fac2
                end do
             end do
             h % energy = h % energy + hk(it,ik) % energy
             h % delfs2(:) = h % delfs2(:) + hk(it,ik) % delfs2(:)
          end do
       end do
       deallocate (wgt)
    end if

    deallocate (tot)

!!
!! Put g, gnew back to their usual meanings
!!
    call g_adjust (g,    phi,    bpar,    -fphi, -fbpar)
    call g_adjust (gnew, phinew, bparnew, -fphi, -fbpar)

  end subroutine get_heat
!==============================================================================

  subroutine reset_init

    use dist_fn_arrays, only: gnew, g
    initializing  = .true.
    initialized = .false.
    
    a = 0. ;  b = 0. ; r = 0. ; ainv = 0.
    g = 0. ; gnew = 0.  ;  g0 = 0.


  end subroutine reset_init

  subroutine reset_physics

    call init_wstar

  end subroutine reset_physics

!>MAB
  subroutine get_verr (errest, erridx, phi, bpar)
     
    use mp, only: proc0
    use le_grids, only: integrate_test, integrate_species
    use le_grids, only: eint_error, lint_error
    use le_grids, only: nlambda, negrid
    use egrid, only: x0
    use theta_grid, only: ntgrid
    use kgrids, only: nakx, naky, aky, akx
    use species, only: nspec, spec
    use dist_fn_arrays, only: gnew, aj0, vpa
    use run_parameters, only: fphi, fapar, fbpar, beta
    use agk_layouts, only: g_lo
    use nonlinear_terms, only: nonlin

    integer :: ig, it, ik, is, il, ipt, iglo, isgn

    integer, dimension (:,:), intent (out) :: erridx
    real, dimension (:,:), intent (out) :: errest
    complex, dimension (-ntgrid:,:,:), intent (in) :: phi, bpar

    real, dimension (:), allocatable :: wgt
    real, dimension (:), allocatable :: errtmp
    integer, dimension (:), allocatable :: idxtmp
    real, dimension (:,:), allocatable, save :: kmax
    complex, dimension (:,:,:), allocatable :: phi_app, apar_app
    complex, dimension (:,:,:,:), allocatable :: phi_e, phi_l
    complex, dimension (:,:,:,:), allocatable :: apar_e, apar_l

    real :: gptmp, gdsum, gpsum, gdmax, gpavg, gnsum, gsmax
    real :: errcut_phi, errcut_apar
    integer :: igmax, ikmax, itmax, gpcnt

    allocate(wgt(nspec))
    allocate(errtmp(2))
    allocate(idxtmp(3))

    if (fphi > epsilon(0.0)) then
       allocate(phi_app(-ntgrid:ntgrid,nakx,naky))
       allocate(phi_e(-ntgrid:ntgrid,nakx,naky,negrid-1))
       allocate(phi_l(-ntgrid:ntgrid,nakx,naky,nlambda))
    end if

    if (fapar > epsilon(0.0)) then
       allocate(apar_app(-ntgrid:ntgrid,nakx,naky))
       allocate(apar_e(-ntgrid:ntgrid,nakx,naky,negrid-1))
       allocate(apar_l(-ntgrid:ntgrid,nakx,naky,nlambda))
    end if

! first call to g_adjust converts gyro-averaged dist. fn. (g)
! into nonadiabatic part of dist. fn. (h)

    call g_adjust (gnew, phi, bpar, fphi, fbpar)

! take gyro-average of h at fixed total position (not g.c. position)
    if (fphi > epsilon(0.0)) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          do isgn = 1, 2
             g0(:,isgn,iglo) = aj0(iglo)*gnew(:,isgn,iglo)
          end do
       end do

       wgt = spec%z*spec%dens

       call integrate_species (g0, wgt, phi_app)

!    call integrate_test (g0, wgt, phi_app, istep)  ! only around for testing

! integrates dist fn of each species over v-space
! after dropping an energy grid point and returns
! phi_e, which contains the integral approximations
! to phi for each point dropped

       call eint_error (g0, wgt, phi_e)

! integrates dist fn of each species over v-space
! after dropping a lambda grid point and returns phi_l.
! phi_l contains nlambda approximations for the integral over lambda that
! come from dropping different pts from the gaussian quadrature grid

       call lint_error (g0, wgt, phi_l)

    end if

    if (fapar > epsilon(0.0)) then
       do iglo = g_lo%llim_proc, g_lo%ulim_proc
          do isgn = 1, 2
             g0(:,isgn,iglo) = aj0(iglo)*vpa(isgn,iglo)*gnew(:,isgn,iglo)
          end do
       end do
       
       wgt = 2.0*beta*spec%z*spec%dens*sqrt(spec%temp/spec%mass)
       call integrate_species (g0, wgt, apar_app)

!    call integrate_test (g0, wgt, apar_app, istep)  ! only around for testing

       call eint_error (g0, wgt, apar_e)
       call lint_error (g0, wgt, apar_l)

    end if

! second call to g_adjust converts from h back to g

    call g_adjust (gnew, phi, bpar, -fphi, -fbpar)

    if (.not. allocated(kmax)) then
       allocate (kmax(nakx, naky))
       do ik = 1, naky
          do it = 1, nakx
             kmax(it,ik) = max(akx(it),aky(ik))
          end do
       end do
    end if
    
    errest = 0.0
    erridx = 0
    
    if (fphi > epsilon(0.0)) then
       errcut_phi = 0.0
       
       do ig = -ntgrid, ntgrid
          errcut_phi = max(errcut_phi, maxval(cabs(phi_app(ig,:,:))*kmax))
       end do
       errcut_phi = errcut_phi/100
       
       call estimate_error (phi_app, phi_e, kmax, errcut_phi, errtmp, idxtmp)
       errest(1,:) = errtmp
       erridx(1,:) = idxtmp
       
       call estimate_error (phi_app, phi_l, kmax, errcut_phi, errtmp, idxtmp)
       errest(2,:) = errtmp
       erridx(2,:) = idxtmp

    end if
    
    if (fapar > epsilon(0.0)) then
       errcut_apar = 0.0
       do ig = -ntgrid, ntgrid
          errcut_apar = max(errcut_apar, maxval(cabs(apar_app(ig,:,:))*kmax))
       end do
       errcut_apar = errcut_apar/100
       
       call estimate_error (apar_app, apar_e, kmax, errcut_apar, errtmp, idxtmp)
       errest(3,:) = errtmp
       erridx(3,:) = idxtmp
       
       call estimate_error (apar_app, apar_l, kmax, errcut_apar, errtmp, idxtmp)
       errest(4,:) = errtmp
       erridx(4,:) = idxtmp
    end if

    deallocate (wgt, errtmp, idxtmp)
    if (fphi > epsilon(0.0)) deallocate(phi_app, phi_e, phi_l)
    if (fapar > epsilon(0.0)) deallocate(apar_app, apar_e, apar_l)

  end subroutine get_verr

  subroutine estimate_error (app1, app2, kmax, errcut, errest, erridx)

    use kgrids, only: naky, nakx
    use theta_grid, only: ntgrid

    implicit none

    complex, dimension (-ntgrid:,:,:), intent (in) :: app1
    complex, dimension (-ntgrid:,:,:,:), intent (in) :: app2
    real, dimension (:,:), intent (in) :: kmax
    real, intent (in) :: errcut
    real, dimension (:), intent (out) :: errest
    integer, dimension (:), intent (out) :: erridx

    integer :: ik, it, ig, ipt
    integer :: igmax, ikmax, itmax, gpcnt
    real :: gdsum, gdmax, gpavg, gnsum, gsmax, gpsum, gptmp

    igmax = 0; ikmax = 0; itmax = 0
    gdsum = 0.0; gdmax = 0.0; gpavg = 0.0; gnsum = 0.0; gsmax = 0.0

    do ik = 1, naky
       do it = 1, nakx
          do ig=-ntgrid,ntgrid
             gpcnt = 0; gpsum = 0.0
             if ((kmax(it,ik)*cabs(app1(ig,it,ik)) > errcut) .and. &
                  (kmax(it,ik)*cabs(app1(ig,it,ik)) > 10*epsilon(0.0))) then
                do ipt=1,size(app2(0,1,1,:))
                      
                   gptmp = kmax(it,ik)*cabs(app1(ig,it,ik) - app2(ig,it,ik,ipt))
                   gpsum = gpsum + gptmp
                   gpcnt = gpcnt + 1
                      
                end do
                   
                gpavg = gpsum/gpcnt
                   
                if (gpavg > gdmax) then
                   igmax = ig
                   ikmax = ik
                   itmax = it
                   gdmax = gpavg
                   gsmax = kmax(it,ik)*cabs(app1(ig,it,ik))
                end if
                   
                gnsum = gnsum + gpavg
                gdsum = gdsum + kmax(it,ik)*cabs(app1(ig,it,ik))

             end if

          end do
       end do
    end do
       
    gdmax = gdmax/gsmax
       
    erridx(1) = igmax
    erridx(2) = ikmax
    erridx(3) = itmax
    errest(1) = gdmax
    errest(2) = gnsum/gdsum

  end subroutine estimate_error
!<MAB

! TT> removed arguments phi and bpar and just use them from fields_arrays
!  subroutine write_f (last, phi, bpar)   ! added phi, bpar - MAB
  subroutine write_f (last)
! <TT

    use mp, only: proc0, send, receive
    use file_utils, only: open_output_file, close_output_file, get_unused_unit
    use agk_layouts, only: g_lo, ik_idx, it_idx, is_idx, il_idx, ie_idx
    use agk_layouts, only: idx_local, proc_id
    use le_grids, only: al, e, negrid, nlambda
    use theta_grid, only: ntgrid
    use agk_time, only: time
    use dist_fn_arrays, only: gnew  ! changed from g to gnew - MAB
    use run_parameters, only: fphi, fbpar   ! MAB
! TT>
    use fields_arrays, only: phinew, bparnew
! <TT

    integer :: iglo, ik, it, is
    integer :: ie, il, ig
    integer, save :: unit
    real :: vpa, vpe
    complex, dimension(2) :: gtmp
    logical :: first = .true.
    logical, intent(in)  :: last 

! TT>
!    complex, dimension (-ntgrid:,:,:), intent (in) :: phi, bpar   ! MAB
! <TT

! xpts and ypts are only temporary additions so that Matlab script works
! in current form (designed for gs2)
    real, dimension (:), allocatable, save :: xpts, ypts

    if (.not. allocated(xpts)) allocate(xpts(negrid))
    if (.not. allocated(ypts)) allocate(ypts(nlambda))

    xpts = 0.0
    ypts = 0.0

    if (proc0) then
       if (first) then 
          call get_unused_unit (unit)
          call open_output_file (unit, ".dist")
          write (unit,*) negrid*nlambda
          first = .false.
       end if
    endif

! TT>
!    call g_adjust (gnew, phi, bpar, fphi, fbpar)
    call g_adjust (gnew, phinew, bparnew, fphi, fbpar)
! <TT

    do iglo = g_lo%llim_world, g_lo%ulim_world
       ik = ik_idx(g_lo, iglo) ; if (ik /= 1) cycle
       it = it_idx(g_lo, iglo) ; if (it /= 1) cycle
       is = is_idx(g_lo, iglo) !; if (is /= 1) cycle
       ie = ie_idx(g_lo, iglo) 
       ig = 0
       il = il_idx(g_lo, iglo)
       if (idx_local (g_lo, ik, it, il, ie, is)) then
          if (proc0) then 
             gtmp = gnew(ig,:,iglo)  ! replaced g with gnew - MAB
          else
             call send (gnew(ig,:,iglo), 0)   ! replaced g with gnew - MAB
          endif
       else if (proc0) then
          call receive (gtmp, proc_id(g_lo, iglo))
       endif
       if (proc0) then
          vpa = sqrt(e(ie,is)*max(0.0, 1.0-al(il)))
          vpe = sqrt(e(ie,is)*al(il))
!             write (unit, "(8(1x,e12.6))") vpe,vpa,gtmp(1),-vpa,gtmp(2)             
!          gtmp = gtmp* exp(-e(ie,is)) ! can be a problem for large ecut
!          write (unit, "(8(1x,e12.6),1x,i2)") vpe,vpa,gtmp(1),-vpa,gtmp(2),time,is
! changed to work with Matlab script for generating 2D plots of g(v)
          write (unit, "(8(1x,e12.6))") vpa, vpe, e(ie,is), al(il), &
               xpts(ie), ypts(il), real(gtmp(1)), real(gtmp(2))
       end if
    end do
    if (proc0) write (unit, *)
    if (last .and. proc0) call close_output_file (unit)
    
! TT>
!    call g_adjust (gnew, phi, bpar, -fphi, -fbpar)
    call g_adjust (gnew, phinew, bparnew, -fphi, -fbpar)
! <TT

  end subroutine write_f
!------------------------------------------------------------------------------
! Velocity Plane Analysis---GGH 23JAN08
!------------------------------------------------------------------------------
! Routine to output slices of velocity space
! For each (it,ik) Fourier mode, it produces a separate file at ig=0 plane
!     for g       runname.vpx(it)y(ik)g
!     for h       runname.vpx(it)y(ik)h
  subroutine write_vp 
    use mp, only: proc0, send, receive, sum_reduce
    use le_grids, only: al, e, negrid, nlambda
    use species, only: nspec
    use kgrids, only: nakx, naky
    use nonlinear_terms, only: nonlin
    use file_utils, only: open_output_file, close_output_file, get_unused_unit
    use dist_fn_arrays, only: gnew  
    use run_parameters, only: fphi, fbpar
    use fields_arrays, only: phinew, bparnew
    use agk_layouts, only: idx, proc_id, idx_local, g_lo
    implicit none
    !Local
    integer :: inn             !1=g, 2=h
    integer :: it              !nakx index
    integer :: ik              !naky index
    integer :: is              !nspec index
    integer :: ie              !negrid index
    integer :: ig              !ntgrid index
    integer :: isgn            !sign of vpar index
    integer :: il              !nlambda index
    integer :: iglo            !iglo g layout index
    integer :: iproc           !Processor number for current meshpoint
    complex, pointer, dimension(:,:,:) :: gv  !v-plane of dist function
    complex, pointer, dimension(:) :: gv2  !v-plane of dist function
    real :: vpar,vperp         !Parallel and perpendicular velocity
    character(10) :: suffix    !Output filename suffix  .vpx##y##
    integer :: unit           !Output File unitnumber
    complex, dimension(2) :: gtmp   !Temporary g (isgn=1,2)

    !Allocate a variable  to contain the slice in velocity space for the given mode
    if (proc0) write(*,'(a,i4,a,i3,a,i2,a)')'Attempting to allocate a complex variable of size (', 2*nlambda, ',', negrid, ',', nspec, ')'
    allocate(gv(1:2*nlambda,1:negrid,1:nspec)); gv=0.
    allocate(gv2(1:2*nlambda*negrid*nspec)); gv2=0.
    if (proc0) write(*,'(a)')'Allocation successful.'

    !Do analysis both for g (inn=1) and h (inn=2)
    do inn=1,2
       if (inn .eq. 2) call g_adjust (gnew, phinew, bparnew, fphi, fbpar)

    !Loop over all Fourier modes to get v-space slice for each one===================
    do ik=1,naky
       do it=1,nakx
          !Skip the constant (0,0) mode if nonlinear run
          if (nonlin .and. it == 1 .and. ik == 1) cycle
          !DEBUG
          if (proc0) write(*,'(a,i2,a,i2,a)')'Analyzing Fourier mode (it,ik)= (',it,',',ik,')'

          !Get slice in v-space-------------------------------------------------------
          if (.false.) then  !Individual send and receive (INEFFICIENT)
          !Loop over species, energy, and pitch angle to assemble data for slice
          do is=1,nspec
             do ie=1,negrid
                do il=1,nlambda
                   !Choose a single point along z
                   ig= 0    !Midplane for the moment (ig runs from -ntgrid to ntgrid)
                   !For this data point (ik,it,is,ie,il) find processor
                   iglo=idx(g_lo,ik,it,il,ie,is)    !Get iglo
                   iproc=proc_id(g_lo,iglo)         !Get proc number for this iglo
                   !If local, get data point and send to gtmp on proc0 
                   if (idx_local (g_lo, ik, it, il, ie, is)) then
                      if (proc0) then 
                         gtmp = gnew(ig,:,iglo)  
                      else
                         call send (gnew(ig,:,iglo), 0)  
                      endif
                   !Receive data point if proc0
                   else if (proc0) then
                      call receive (gtmp, iproc)
                   endif
                   !Put the values into the correct place in gv
                   if (proc0) then
                      gv(ivpar(1,il),ie,is)=cmplx(gtmp(1))
                      gv(ivpar(2,il),ie,is)=cmplx(gtmp(2))
                   endif
                enddo
             enddo
          enddo
          else !Sum_reduce method (EFFICIENT)
          gv=0.  !Initialize slice variable
          !Loop over species, energy, and pitch angle to assemble data for slice
          do is=1,nspec
             do ie=1,negrid
                do il=1,nlambda
                   !Choose a single point along z
                   ig= 0    !Midplane for the moment (ig runs from -ntgrid to ntgrid)
                   !For this data point (ik,it,is,ie,il) find processor
                   iglo=idx(g_lo,ik,it,il,ie,is)    !Get iglo
                   !If local, put data point and send to gtmp on proc0 
                   if (idx_local (g_lo, ik, it, il, ie, is)) then
                      gtmp = gnew(ig,:,iglo)  
                      gv(ivpar(1,il),ie,is)=cmplx(gtmp(1))
                      gv(ivpar(2,il),ie,is)=cmplx(gtmp(2))
                   endif
                enddo
             enddo
          enddo
          !Sum_reduce to proc0 to get all values of array
          !Reshape to 1-D array
          gv2=reshape(gv,(/ 2*nlambda*negrid*nspec /))
          !Call sum_reduce on 1-D array to get all variables to proc0
          call sum_reduce(gv2,0)
          !Reshape back to 3-D array
          gv=reshape(gv2,(/ 2*nlambda,negrid,nspec /))
          endif


          !Output data to file runname.vpx##y##g or  .vpx##y##h   ------------------
          if (proc0) then
            !Create output file suffix
             if (inn .eq. 1) then
                write(suffix,'(a4,i2.2,a1,i2.2,a1)')'.vpx',it,'y',ik,'g'
             elseif (inn .eq. 2) then
                write(suffix,'(a4,i2.2,a1,i2.2,a1)')'.vpx',it,'y',ik,'h'
             endif
             !DEBUG
             write(*,'(a,i2,a,i2,a,a)')'Output for Fourier mode (it,ik)= (',it,',',ik,') to ',suffix
 
             !Open file runname.vpx##y##(g or h)
             call get_unused_unit (unit)
             call open_output_file (unit,suffix)
             
             do is=1,nspec
                do ie=1,negrid
                   isgn=2
                   do il=1,nlambda
                      !Calculate vpar and vperp for this value
                      vpar  = -sqrt(e(ie,is)*max(0.0, 1.0-al(il)))
                      vperp = sqrt(e(ie,is)*al(il))
                      write(unit,'(4es14.6,4i4)')vpar,vperp,gv(ivpar(1,il),ie,is),is,ie,isgn,il
                   enddo
                   isgn=1
                   do il=nlambda,1,-1
                      !Calculate vpar and vperp for this value
                      vpar  = sqrt(e(ie,is)*max(0.0, 1.0-al(il)))
                      vperp = sqrt(e(ie,is)*al(il))
                      write(unit,'(4es14.6,4i4)')vpar,vperp,gv(ivpar(1,il),ie,is),is,ie,isgn,il
                   enddo
                   write(unit,'(a)')' '
                enddo
             enddo

             !Close file
             call close_output_file (unit)
          end if
          
       enddo
    enddo !END Loop over all Fourier modes===========================================
    if (inn .eq. 2) call g_adjust (gnew, phinew, bparnew, -fphi, -fbpar)
    enddo !Loop over inn (1=g, 2=h)

    !Deallocate
    if (proc0) deallocate(gv,gv2)

  end subroutine write_vp
!------------------------------------------------------------------------------
! ivpar  GGH 23JAN08
!------------------------------------------------------------------------------
! Function to get sorted single index for the pitch angle and sign for write_vp
  integer function ivpar(isgn,il)
    use le_grids, only: nlambda
    implicit none
    integer :: isgn            !sign of vpar index
    integer :: il              !nlambda index

    if (isgn .eq. 1) then
       ivpar=nlambda+il
    else
       ivpar=nlambda-il+1
    endif
  end function ivpar
!------------------------------------------------------------------------------
! This subroutine only returns epar correctly for linear runs.
  subroutine get_epar (phi, apar, phinew, aparnew, epar)
    use theta_grid, only: ntgrid, delthet, gradpar
    use run_parameters, only: fphi, fapar
    use agk_time, only: dtime
    use kgrids, only: naky, nakx
    complex, dimension(-ntgrid:,:,:) :: phi, apar, phinew, aparnew, epar
    complex :: phi_m, apar_m

    integer :: ig, ik, it

    do ik = 1, naky
       do it = 1, nakx
          do ig = -ntgrid, ntgrid-1
             ! ignoring decentering in time and space for now
             phi_m = 0.5*(phi(ig+1,it,ik)-phi(ig,it,ik) + &
                  phinew(ig+1,it,ik)-phinew(ig,it,ik))*fphi
             apar_m = 0.5*(aparnew(ig+1,it,ik)+aparnew(ig,it,ik) & 
                  -apar(ig+1,it,ik)-apar(ig,it,ik))*fapar
             
             epar(ig,it,ik) = -phi_m/delthet*gradpar - apar_m/dtime
          end do
       end do
    end do    

  end subroutine get_epar

  subroutine dot (a, anew, adot, fac)

! Get the time derivative of a field.
! 
    use agk_time, only: dtime
    use kgrids, only: naky, nakx
    use theta_grid, only: ntgrid

    implicit none
    complex, intent (in), dimension (-ntgrid:,:,:) :: a, anew
    complex, intent (out), dimension (-ntgrid:,:,:) :: adot
    real, intent (in) :: fac
    real :: dtinv
    integer :: ig, it, ik

    dtinv = 1./dtime
    do ik=1,naky
       do it=1,nakx
          do ig=-ntgrid,ntgrid
             adot(ig,it,ik) = fac*(anew(ig,it,ik) - a(ig,it,ik))*dtinv
          end do
       end do
    end do
    
  end subroutine dot

end module dist_fn

