module fields
  use fields_arrays, only: phi, apar, bpar, phinew, aparnew, bparnew
  use fields_arrays, only: phi_ext, apar_ext

  implicit none

  public :: init_fields  ! initialize this module
  public :: advance      ! do a full time step
  public :: phinorm      ! diagnostic variable of no intrinsic value
  public :: phi, apar, bpar, phinew, aparnew, bparnew  ! field variables at old and new timestep
  public :: reset_init   ! Sets fields to zero.  Useful for certain kinds of restarts.

  private

  logical :: initialized = .false.

contains

  subroutine init_fields
    use mp, only: proc0
    use theta_grid, only: init_theta_grid
    use run_parameters, only: init_run_parameters
    use dist_fn, only: init_dist_fn, ginit
    use init_g, only: init_init_g
    use fields_implicit, only: init_fields_implicit, init_phi_implicit
    use nonlinear_terms, only: nl_finish_init => finish_init
    use antenna, only: init_antenna
    implicit none
    logical :: restarted

    if (initialized) return  ;  initialized = .true.

    call init_theta_grid       ! Set up grid along the field line.
    call init_init_g           ! Initialize module that handles initial conditions
    call init_run_parameters   ! Get basic inputs, set up perpendicular grids.
    call init_dist_fn          ! Make sure code is ready to calculate with distribution function
    call allocate_arrays       ! Allocate memory as necessary
    call init_fields_implicit  ! Calculate matrices for implicit soln of Maxwell eqns.

! Turn on nonlinear terms if necessary.
    call nl_finish_init

    call ginit (restarted)
    call init_antenna
    if (restarted) return

    call init_phi_implicit  ! actually: set up fields for first time step.
    
  end subroutine init_fields

  subroutine allocate_arrays
    use theta_grid, only: ntgrid
    use kgrids, only: naky, nakx
    implicit none
    logical :: alloc = .true.

    if (alloc) then
       allocate (     phi (-ntgrid:ntgrid,nakx,naky))
       allocate (    apar (-ntgrid:ntgrid,nakx,naky))
       allocate (    bpar (-ntgrid:ntgrid,nakx,naky))
       allocate (  phinew (-ntgrid:ntgrid,nakx,naky))
       allocate ( aparnew (-ntgrid:ntgrid,nakx,naky))
       allocate (bparnew  (-ntgrid:ntgrid,nakx,naky))
       allocate (apar_ext (-ntgrid:ntgrid,nakx,naky))
    endif

    phi  = 0. ; phinew  = 0.
    apar = 0. ; aparnew = 0.
    bpar = 0. ; bparnew = 0.

    apar_ext = 0.

    alloc = .false.
  end subroutine allocate_arrays

  subroutine advance (istep)
    use fields_implicit, only: advance_implicit
    implicit none
    integer, intent (in) :: istep

! Could call different algorithm here...e.g., something explicit.

    call advance_implicit (istep)

  end subroutine advance

  subroutine phinorm (phitot)
    use theta_grid, only: delthet
    use kgrids, only: naky, nakx
    use constants
    implicit none
    real, dimension (:,:), intent (out) :: phitot
    integer :: ik, it

! GS2 reported the square of this value.  Only used for rough diagnostic purposes.

    do ik = 1, naky
       do it = 1, nakx
          phitot(it,ik) = 0.5/pi &
           *(sum((cabs(phinew(:,it,ik)) + cabs(aparnew(:,it,ik)) &
                  + cabs(bparnew(:,it,ik)))*delthet))
       end do
    end do

  end subroutine phinorm

  subroutine reset_init

    initialized  = .false.
    phi  = 0.  ;  phinew  = 0.
    apar = 0.  ;  aparnew = 0.  ! GS2 did not have these statements!
    bpar = 0.  ;  bparnew = 0.  ! GS2 did not have these statements!

  end subroutine reset_init

end module fields
