! Solve the Poisson equation Laplace U = rho

module potential
  implicit none
  
  integer, parameter :: rk = selected_real_kind(15, 100)
  
  ! Grid size
  integer,  parameter :: n = 100
  real(rk), parameter :: h = 1.0_rk / n
  
  ! Number of iterations
  integer, parameter :: niters = 100
  
  ! Step size factor
  real(rk), parameter :: alpha = 0.24_rk
  
  real(rk), allocatable :: rho(:,:) ! density
  real(rk), allocatable :: pot(:,:) ! potential
  real(rk), allocatable :: res(:,:) ! residual
  
contains
  
  ! Calculate the residual everywhere.
  ! This evaluates how much the Poisson equation is violated.
  subroutine residual
    integer :: i,j
    real(rk) :: pi
    pi = acos(-1.0_rk)
    do j=1,n-1
       do i=1,n-1
          res(i,j) = &
               + (pot(i-1,j) - 2*pot(i,j) + pot(i+1,j)) / h**2 &
               + (pot(i,j-1) - 2*pot(i,j) + pot(i,j+1)) / h**2 &
               - 4 * pi * rho(i,j)
       end do
    end do
  end subroutine residual
  
  subroutine initial
    ! Set the density to zero everywhere except in the centre.
    ! Ensure the total mass is 1.
    rho(:,:) = 0
    rho(n/2,n/2) = n**2
    
    ! Initialise the potential and the residual to zero
    pot(:,:) = 0
    res(:,:) = 0
  end subroutine initial
  
  ! Take one iteration step.
  ! Add a fraction of the residual to the potential, which will reduce
  ! the residual.
  subroutine step
    integer :: i,j
    do j=0,n
       do i=0,n
          pot(i,j) = pot(i,j) + alpha * h**2 * res(i,j)
       end do
    end do
  end subroutine step
  
end module potential



program calc_potential
  use potential
  implicit none
  
  integer  :: iter              ! Iteration count
  real(rk) :: res_l2            ! L2 norm of the residual
  
  real(rk) :: t0, t1
  
  allocate(rho(0:n,0:n))
  allocate(pot(0:n,0:n))
  allocate(res(0:n,0:n))
  
  print '("Solving for potential")'
  print '("   using n=",i0,", niters=",i0)', n, niters
  print '()'
  
  ! Initialise
  call initial
  ! Evaluate the residual
  call residual
  ! Output the residual
  res_l2 = sqrt(sum(res(:,:)**2) / (n-1)**2)
  print '("Iteration ",i8, "   L2[residual] ",g25.15)', 0, res_l2
  
  ! Iterate
  do iter = 1, niters
     ! Take an iteration step
     call step
     ! Evaluate the residual
     call residual
     ! Output the residual, but not too often
     if (mod(iter, niters/10) == 0) then
        res_l2 = sqrt(sum(res(:,:)**2) / (n-1)**2)
        print '("Iteration ",i8, "   L2[residual] ",g25.15)', iter, res_l2
     end if
  end do
end program calc_potential
